"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DocumentSharingLocation = void 0;
const ExtensionMethods_1 = require("../ExtensionMethods");
const EwsLogging_1 = require("../Core/EwsLogging");
const XmlElementNames_1 = require("../Core/XmlElementNames");
/**
 * Represents a sharing location.
 * @sealed
 */
class DocumentSharingLocation {
    /**
     * Initializes a new instance of the **DocumentSharingLocation** class.
     */
    constructor() {
        /**
         * The URL of the web service to use to manipulate documents at the sharing location.
         */
        this.serviceUrl = null;
        /**
         * The URL of the sharing location (for viewing the contents in a web browser).
         */
        this.locationUrl = null;
        /**
         * The display name of the location.
         */
        this.displayName = null;
        /**
         * The set of file extensions that are allowed at the location.
         */
        this.supportedFileExtensions = [];
        /**
         * Indicates whether external users (outside the enterprise/tenant) can view documents at the location.
         */
        this.externalAccessAllowed = false;
        /**
         * Indicates whether anonymous users can view documents at the location.
         */
        this.anonymousAccessAllowed = false;
        /**
         * Indicates whether the user can modify permissions for documents at the location.
         */
        this.canModifyPermissions = false;
        /**
         * Indicates whether this location is the user's default location. This will generally be their My Site.
         */
        this.isDefault = false;
    }
    // ref: no need to have setter for all properties as all of them are private, can directly be assigned to private variables.
    /**
     * Gets the URL of the web service to use to manipulate documents at the sharing location.
     */
    get ServiceUrl() {
        return this.serviceUrl;
    }
    /**
     * Gets the URL of the sharing location (for viewing the contents in a web browser).
     */
    get LocationUrl() {
        return this.locationUrl;
    }
    /**
     * Gets the display name of the location.
     */
    get DisplayName() {
        return this.displayName;
    }
    /**
     * Gets the space-separated list of file extensions that are allowed at the location.
     * @remarks Example:  "docx pptx xlsx"
     */
    get SupportedFileExtensions() {
        return this.supportedFileExtensions;
    }
    /**
     * Gets a flag indicating whether external users (outside the enterprise/tenant) can view documents at the location.
     */
    get ExternalAccessAllowed() {
        return this.externalAccessAllowed;
    }
    /**
     * Gets a flag indicating whether anonymous users can view documents at the location.
     */
    get AnonymousAccessAllowed() {
        return this.anonymousAccessAllowed;
    }
    /**
     * Gets a flag indicating whether the user can modify permissions for documents at the location.
     * @remarks This will be true for the user's "My Site," for example. However, documents at team and project sites will typically be ACLed by the site owner, so the user will not be able to modify permissions. This will most likely by false even if the caller is the owner, to avoid surprises. They should go to SharePoint to modify permissions for team and project sites.
     */
    get CanModifyPermissions() {
        return this.canModifyPermissions;
    }
    /**
     * Gets a flag indicating whether this location is the user's default location.  This will generally be their My Site.
     */
    get IsDefault() {
        return this.isDefault;
    }
    /**
     * @internal Loads DocumentSharingLocation instance.
     *
     * @param   {any} jsObject  Json Object converted from XML.
     */
    static LoadFromXmlJsObject(jsObject) {
        EwsLogging_1.EwsLogging.Assert(false, "DocumentSharingLocation.LoadFromXmlJsObject", "Please provide feedback if this is successful or failed.", true);
        const location = new DocumentSharingLocation();
        location.serviceUrl = jsObject[XmlElementNames_1.XmlElementNames.ServiceUrl];
        location.locationUrl = jsObject[XmlElementNames_1.XmlElementNames.LocationUrl];
        location.displayName = jsObject[XmlElementNames_1.XmlElementNames.DisplayName];
        location.supportedFileExtensions = [];
        const supportedFileExtensions = jsObject[XmlElementNames_1.XmlElementNames.SupportedFileExtensions];
        let extensions = supportedFileExtensions[XmlElementNames_1.XmlElementNames.FileExtension];
        if (!Array.isArray(extensions)) {
            extensions = [extensions];
        }
        for (let i = 0; i < extensions.length; i++) {
            location.supportedFileExtensions.push(extensions[i]);
        }
        jsObject[XmlElementNames_1.XmlElementNames.SupportedFileExtensions];
        location.externalAccessAllowed = ExtensionMethods_1.Convert.toBool(jsObject[XmlElementNames_1.XmlElementNames.ExternalAccessAllowed]);
        location.anonymousAccessAllowed = ExtensionMethods_1.Convert.toBool(jsObject[XmlElementNames_1.XmlElementNames.AnonymousAccessAllowed]);
        location.canModifyPermissions = ExtensionMethods_1.Convert.toBool(jsObject[XmlElementNames_1.XmlElementNames.CanModifyPermissions]);
        location.isDefault = ExtensionMethods_1.Convert.toBool(jsObject[XmlElementNames_1.XmlElementNames.IsDefault]);
        return location;
    }
}
exports.DocumentSharingLocation = DocumentSharingLocation;
//# sourceMappingURL=DocumentSharingLocation.js.map