"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GetDomainSettingsResponse = void 0;
const AltDictionary_1 = require("../../AltDictionary");
const DomainSettingError_1 = require("../DomainSettingError");
const DomainSettingName_1 = require("../../Enumerations/DomainSettingName");
const EwsLogging_1 = require("../../Core/EwsLogging");
const ExtensionMethods_1 = require("../../ExtensionMethods");
const XmlElementNames_1 = require("../../Core/XmlElementNames");
const AutodiscoverResponse_1 = require("./AutodiscoverResponse");
/**
 * Represents the response to a GetDomainSettings call for an individual domain.
 * @sealed
 */
class GetDomainSettingsResponse extends AutodiscoverResponse_1.AutodiscoverResponse {
    /**
     * Initializes a new instance of the **GetDomainSettingsResponse** class.
     */
    constructor() {
        super();
        this.domain = null;
        this.redirectTarget = null;
        this.settings = null;
        this.domainSettingErrors = null;
        this.domain = ExtensionMethods_1.StringHelper.Empty;
        this.settings = new AltDictionary_1.Dictionary((value) => value.toString());
        this.domainSettingErrors = [];
    }
    /**
     * Gets the domain this response applies to.
     */
    get Domain() {
        return this.domain;
    }
    /** @internal */
    set Domain(value) {
        this.domain = value;
    }
    /**
     * Gets the redirectionTarget (URL or email address)
     */
    get RedirectTarget() {
        return this.redirectTarget;
    }
    /**
     * Gets the requested settings for the domain.
     */
    get Settings() {
        return this.settings;
    }
    /**
     * Gets error information for settings that could not be returned.
     */
    get DomainSettingErrors() {
        return this.domainSettingErrors;
    }
    /**
     * Loads the domain setting errors.
     *
     * @param   {any} jsObject  Json Object converted from XML.
     */
    LoadDomainSettingErrorsFromXmlJsObject(jsObject) {
        var errors = undefined;
        if ((0, ExtensionMethods_1.isNullOrUndefined)(jsObject) || (0, ExtensionMethods_1.isNullOrUndefined)(jsObject[XmlElementNames_1.XmlElementNames.DomainSettingError]))
            return;
        if (Array.isArray(jsObject[XmlElementNames_1.XmlElementNames.DomainSettingError]))
            errors = jsObject[XmlElementNames_1.XmlElementNames.DomainSettingError];
        else
            errors = [jsObject[XmlElementNames_1.XmlElementNames.DomainSettingError]];
        for (var i = 0; i < errors.length; i++) {
            var error = new DomainSettingError_1.DomainSettingError();
            error.LoadFromXmlJsObject(errors[i]);
            this.DomainSettingErrors.push(error);
        }
    }
    /**
     * @internal Loads setting from XML jsObject.
     *
     * @param   {any} jsObject  Json Object converted from XML.
     */
    LoadDomainSettingsFromXmlJsObject(obj) {
        var settings = undefined;
        if (typeof (obj[XmlElementNames_1.XmlElementNames.DomainSetting]) === 'undefined')
            return;
        if (Object.prototype.toString.call(obj[XmlElementNames_1.XmlElementNames.DomainSetting]) === "[object Array]")
            settings = obj[XmlElementNames_1.XmlElementNames.DomainSetting];
        else
            settings = [obj[XmlElementNames_1.XmlElementNames.DomainSetting]];
        for (var i = 0; i < settings.length; i++) {
            var setting = settings[i];
            var settingClass = setting["type"];
            switch (settingClass) {
                case XmlElementNames_1.XmlElementNames.DomainStringSetting:
                    this.ReadSettingFromXmlJsObject(setting);
                    break;
                default:
                    EwsLogging_1.EwsLogging.Assert(false, "GetUserSettingsResponse.LoadUserSettingsFromXml", ExtensionMethods_1.StringHelper.Format("Invalid setting class '{0}' returned", settingClass));
                    break;
            }
        }
    }
    /**
     * @internal Load from XML jsObject.
     *
     * @param   {any} jsObject  Json Object converted from XML.
     */
    LoadFromXmlJsObject(jsObject) {
        super.LoadFromXmlJsObject(jsObject);
        var settingscol = jsObject[XmlElementNames_1.XmlElementNames.DomainSettings];
        this.LoadDomainSettingsFromXmlJsObject(settingscol);
        this.redirectTarget = jsObject[XmlElementNames_1.XmlElementNames.RedirectTarget] || null;
        this.LoadDomainSettingErrorsFromXmlJsObject(jsObject[XmlElementNames_1.XmlElementNames.DomainSettingErrors]);
    }
    /**
     * Reads domain setting from XML jsObject.
     *
     * @param   {any} jsObject  Json Object converted from XML.
     */
    ReadSettingFromXmlJsObject(obj) {
        var name = obj[XmlElementNames_1.XmlElementNames.Name];
        var value = obj[XmlElementNames_1.XmlElementNames.Value];
        // EWS Managed API is broken with AutoDSvc endpoint in RedirectUrl scenario
        var domainSettingName = DomainSettingName_1.DomainSettingName[name]; // EwsUtilities.Parse<UserSettingName>(name);
        if (domainSettingName !== undefined)
            this.Settings.Add(domainSettingName, value);
        else
            EwsLogging_1.EwsLogging.Assert(false, "GetUserSettingsResponse.ReadSettingFromObject", "Unexpected or empty name element in user setting");
    }
}
exports.GetDomainSettingsResponse = GetDomainSettingsResponse;
//# sourceMappingURL=GetDomainSettingsResponse.js.map