"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GetUserSettingsResponse = void 0;
const AlternateMailboxCollection_1 = require("../AlternateMailboxCollection");
const AltDictionary_1 = require("../../AltDictionary");
const DocumentSharingLocationCollection_1 = require("../DocumentSharingLocationCollection");
const EwsLogging_1 = require("../../Core/EwsLogging");
const ProtocolConnectionCollection_1 = require("../ProtocolConnectionCollection");
const ExtensionMethods_1 = require("../../ExtensionMethods");
const UserSettingError_1 = require("../UserSettingError");
const UserSettingName_1 = require("../../Enumerations/UserSettingName");
const WebClientUrlCollection_1 = require("../WebClientUrlCollection");
const XmlElementNames_1 = require("../../Core/XmlElementNames");
const AutodiscoverResponse_1 = require("./AutodiscoverResponse");
/**
 * Represents the response to a GetUsersSettings call for an individual user.
 * @sealed
 */
class GetUserSettingsResponse extends AutodiscoverResponse_1.AutodiscoverResponse {
    /**
     * Initializes a new instance of the **GetUserSettingsResponse** class.
     */
    constructor() {
        super();
        this.smtpAddress = null;
        this.redirectTarget = null;
        this.settings = null;
        this.userSettingErrors = null;
        this.smtpAddress = ExtensionMethods_1.StringHelper.Empty;
        this.settings = new AltDictionary_1.Dictionary(value => value.toString());
        this.userSettingErrors = [];
    }
    /**
     * Gets the SMTP address this response applies to.
     */
    get SmtpAddress() {
        return this.smtpAddress;
    }
    /** @internal */
    set SmtpAddress(value) {
        this.smtpAddress = value;
    }
    /**
     * Gets the redirectionTarget (URL or email address)
     */
    get RedirectTarget() {
        return this.redirectTarget;
    }
    /** @internal */
    set RedirectTarget(value) {
        this.redirectTarget = value;
    }
    /**
     * Gets the requested settings for the user.
     */
    get Settings() {
        return this.settings;
    }
    /** @internal */
    set Settings(value) {
        this.settings = value;
    }
    /**
     * Gets error information for settings that could not be returned.
     */
    get UserSettingErrors() {
        return this.userSettingErrors;
    }
    /** @internal */
    set UserSettingErrors(value) {
        this.userSettingErrors = value;
    }
    /**
     * @internal Load from XML jsObject.
     *
     * @param   {any} jsObject  Json Object converted from XML.
     */
    LoadFromXmlJsObject(jsObject) {
        super.LoadFromXmlJsObject(jsObject);
        var settingscol = jsObject[XmlElementNames_1.XmlElementNames.UserSettings];
        this.LoadUserSettingsFromXmlJsObject(settingscol);
        this.redirectTarget = jsObject[XmlElementNames_1.XmlElementNames.RedirectTarget] || null;
        this.LoadUserSettingErrorsFromXmlJsObject(jsObject[XmlElementNames_1.XmlElementNames.UserSettingErrors]);
    }
    /**
     * Loads the user setting errors.
     *
     * @param   {any} jsObject  Json Object converted from XML.
     */
    LoadUserSettingErrorsFromXmlJsObject(jsObject) {
        var errors = undefined;
        if ((0, ExtensionMethods_1.isNullOrUndefined)(jsObject) || (0, ExtensionMethods_1.isNullOrUndefined)(jsObject[XmlElementNames_1.XmlElementNames.UserSettingError]))
            return;
        if (Array.isArray(jsObject[XmlElementNames_1.XmlElementNames.UserSettingError]))
            errors = jsObject[XmlElementNames_1.XmlElementNames.UserSettingError];
        else
            errors = [jsObject[XmlElementNames_1.XmlElementNames.UserSettingError]];
        for (var i = 0; i < errors.length; i++) {
            var error = new UserSettingError_1.UserSettingError();
            error.LoadFromXmlJsObject(errors[i]);
            this.UserSettingErrors.push(error);
        }
    }
    /**
     * @internal Loads setting from XML jsObject.
     *
     * @param   {any} jsObject  Json Object converted from XML.
     */
    LoadUserSettingsFromXmlJsObject(jsObject) {
        var settings = undefined;
        if (!jsObject || typeof (jsObject[XmlElementNames_1.XmlElementNames.UserSetting]) === 'undefined')
            return;
        if (Object.prototype.toString.call(jsObject[XmlElementNames_1.XmlElementNames.UserSetting]) === "[object Array]")
            settings = jsObject[XmlElementNames_1.XmlElementNames.UserSetting];
        else
            settings = [jsObject[XmlElementNames_1.XmlElementNames.UserSetting]];
        for (var i = 0; i < settings.length; i++) {
            var setting = settings[i];
            var settingClass = setting["type"];
            switch (settingClass) {
                case XmlElementNames_1.XmlElementNames.StringSetting:
                case XmlElementNames_1.XmlElementNames.WebClientUrlCollectionSetting:
                case XmlElementNames_1.XmlElementNames.AlternateMailboxCollectionSetting:
                case XmlElementNames_1.XmlElementNames.ProtocolConnectionCollectionSetting:
                case XmlElementNames_1.XmlElementNames.DocumentSharingLocationCollectionSetting:
                    this.ReadSettingFromXmlJsObject(setting);
                    break;
                default:
                    EwsLogging_1.EwsLogging.Assert(false, "GetUserSettingsResponse.LoadUserSettingsFromXml", ExtensionMethods_1.StringHelper.Format("Invalid setting class '{0}' returned", settingClass));
                    break;
            }
        }
    }
    /**
     * Reads user setting from XML jsObject.
     *
     * @param   {any} jsObject  Json Object converted from XML.
     */
    ReadSettingFromXmlJsObject(jsObject) {
        var name = jsObject[XmlElementNames_1.XmlElementNames.Name];
        var value = jsObject[XmlElementNames_1.XmlElementNames.Value];
        switch (jsObject["type"]) {
            case XmlElementNames_1.XmlElementNames.WebClientUrlCollectionSetting: //.WebClientUrls:
                value = WebClientUrlCollection_1.WebClientUrlCollection.LoadFromXmlJsObject(jsObject[XmlElementNames_1.XmlElementNames.WebClientUrls]);
                break;
            case XmlElementNames_1.XmlElementNames.ProtocolConnectionCollectionSetting: //ProtocolConnections:
                value = ProtocolConnectionCollection_1.ProtocolConnectionCollection.LoadFromXmlJsObject(jsObject[XmlElementNames_1.XmlElementNames.ProtocolConnections]);
                break;
            case XmlElementNames_1.XmlElementNames.AlternateMailboxCollectionSetting: //AlternateMailboxes:
                value = AlternateMailboxCollection_1.AlternateMailboxCollection.LoadFromXmlJsObject(jsObject[XmlElementNames_1.XmlElementNames.AlternateMailboxes]);
                break;
            case XmlElementNames_1.XmlElementNames.DocumentSharingLocationCollectionSetting: //DocumentSharingLocations:
                //debugger;
                EwsLogging_1.EwsLogging.Log("------------DocumentSharingLocationCollection needs test and fix ----------------", true);
                EwsLogging_1.EwsLogging.Log(jsObject, true, true);
                value = DocumentSharingLocationCollection_1.DocumentSharingLocationCollection.LoadFromXmlJsObject(jsObject);
                break;
        }
        // EWS Managed API is broken with AutoDSvc endpoint in RedirectUrl scenario
        var userSettingName = UserSettingName_1.UserSettingName[name]; // EwsUtilities.Parse<UserSettingName>(name);
        if (userSettingName !== undefined)
            this.Settings.Add(userSettingName, value);
        else
            EwsLogging_1.EwsLogging.Assert(false, "GetUserSettingsResponse.ReadSettingFromXml", "Unexpected or empty name element in user setting");
    }
    /**
     * Tries the get the user setting value.
     *
     * @param   {UserSettingName}   setting   The setting.
     * @return  {T}  The setting value.
     */
    GetSettingValue(setting) {
        //public bool TryGetSettingValue<T>(UserSettingName setting, out T value)
        return this.Settings.get(setting);
    }
}
exports.GetUserSettingsResponse = GetUserSettingsResponse;
//# sourceMappingURL=GetUserSettingsResponse.js.map