"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Attachment = void 0;
const ExtensionMethods_1 = require("../ExtensionMethods");
const EwsUtilities_1 = require("../Core/EwsUtilities");
const ExchangeVersion_1 = require("../Enumerations/ExchangeVersion");
const Strings_1 = require("../Strings");
const TypeContainer_1 = require("../TypeContainer");
const XmlAttributeNames_1 = require("../Core/XmlAttributeNames");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const XmlNamespace_1 = require("../Enumerations/XmlNamespace");
const ComplexProperty_1 = require("./ComplexProperty");
/**
 * Represents an attachment to an item.
 */
class Attachment extends ComplexProperty_1.ComplexProperty {
    constructor(ownerOrService) {
        super();
        this.owner = null;
        this.id = null;
        this.name = null;
        this.contentType = null;
        this.contentId = null;
        this.contentLocation = null;
        this.size = 0;
        this.lastModifiedTime = null;
        this.isInline = false;
        this.service = null;
        if (arguments.length === 1 && (ownerOrService === null || ownerOrService instanceof TypeContainer_1.TypeContainer.Item)) {
            this.owner = ownerOrService;
            if (ownerOrService !== null) {
                this.service = this.owner.Service;
            }
            return;
        }
        this.service = ownerOrService;
    }
    /**
     * Gets the Id of the attachment.
     */
    get Id() {
        return this.id;
    }
    set Id(value) {
        this.id = value;
    }
    /***
     * Gets or sets the name of the attachment.
     */
    get Name() {
        return this.name;
    }
    set Name(value) {
        this.SetFieldValue({ getValue: () => this.name, setValue: (updateValue) => { this.name = updateValue; } }, value);
    }
    /**
     * Gets or sets the content type of the attachment.
     */
    get ContentType() {
        return this.contentType;
    }
    set ContentType(value) {
        this.SetFieldValue({ getValue: () => this.contentType, setValue: (updateValue) => { this.contentType = updateValue; } }, value);
    }
    /**
     * Gets or sets the content Id of the attachment. ContentId can be used as a custom way to identify an attachment in order to reference it from within the body of the item the attachment belongs to.
     */
    get ContentId() {
        return this.contentId;
    }
    set ContentId(value) {
        this.SetFieldValue({ getValue: () => this.contentId, setValue: (updateValue) => { this.contentId = updateValue; } }, value);
    }
    /**
     * Gets or sets the content location of the attachment. ContentLocation can be used to associate an attachment with a Url defining its location on the Web.
     */
    get ContentLocation() {
        return this.contentLocation;
    }
    set ContentLocation(value) {
        this.SetFieldValue({ getValue: () => this.contentLocation, setValue: (updateValue) => { this.contentLocation = updateValue; } }, value);
    }
    /**
     * Gets the size of the attachment.
     */
    get Size() {
        EwsUtilities_1.EwsUtilities.ValidatePropertyVersion(this.service, ExchangeVersion_1.ExchangeVersion.Exchange2010, "Size");
        return this.size;
    }
    set Size(value) {
        EwsUtilities_1.EwsUtilities.ValidatePropertyVersion(this.service, ExchangeVersion_1.ExchangeVersion.Exchange2010, "Size");
        this.SetFieldValue({ getValue: () => this.size, setValue: (updateValue) => { this.size = updateValue; } }, value);
    }
    /**
     * Gets the date and time when this attachment was last modified.
     */
    get LastModifiedTime() {
        EwsUtilities_1.EwsUtilities.ValidatePropertyVersion(this.service, ExchangeVersion_1.ExchangeVersion.Exchange2010, "LastModifiedTime");
        return this.lastModifiedTime;
    }
    set LastModifiedTime(value) {
        EwsUtilities_1.EwsUtilities.ValidatePropertyVersion(this.service, ExchangeVersion_1.ExchangeVersion.Exchange2010, "LastModifiedTime");
        this.SetFieldValue({ getValue: () => this.lastModifiedTime, setValue: (updateValue) => { this.lastModifiedTime = updateValue; } }, value);
    }
    /**
     * Gets or sets a value indicating whether this is an inline attachment. Inline attachments are not visible to end users.
     */
    get IsInline() {
        EwsUtilities_1.EwsUtilities.ValidatePropertyVersion(this.service, ExchangeVersion_1.ExchangeVersion.Exchange2010, "IsInline");
        return this.isInline;
    }
    set IsInline(value) {
        EwsUtilities_1.EwsUtilities.ValidatePropertyVersion(this.service, ExchangeVersion_1.ExchangeVersion.Exchange2010, "IsInline");
        this.SetFieldValue({ getValue: () => this.isInline, setValue: (updateValue) => { this.isInline = updateValue; } }, value);
    }
    /**
     * @internal True if the attachment has not yet been saved, false otherwise.
     */
    get IsNew() {
        return ExtensionMethods_1.StringHelper.IsNullOrEmpty(this.Id);
    }
    /**
     * @internal Gets the owner of the attachment.
     */
    get Owner() {
        return this.owner;
    }
    /**
     * @internal Gets the related exchange service.
     */
    get Service() {
        return this.service;
    }
    /**
     * Gets the name of the XML element.
     *
     * @return  {string}      XML element name.
     */
    GetXmlElementName() { console.log("Attachment.ts - GetXmlElementName : Abstract - must implement."); return ExtensionMethods_1.StringHelper.Empty; }
    /**
     * @internal Load the attachment.
     *
     * @param   {BodyType}                      bodyType               Type of the body.
     * @param   {PropertyDefinitionBase[]}      additionalProperties   The additional properties.
     */
    InternalLoad(bodyType, additionalProperties) {
        return this.service.GetAttachment(this, bodyType, additionalProperties);
    }
    //InternalToJson(service: ExchangeService): any { throw new Error("Attachment.ts - InternalToJson : Not implemented."); }
    /**
     * Loads the attachment. Calling this method results in a call to EWS.
     */
    Load() {
        return this.InternalLoad(null, null);
    }
    /**
     * Loads the attachment id from json.
     *
     * @param   {any}   jsonObject   The json object.
     */
    LoadAttachmentIdFromXMLJsObject(jsonObject) {
        this.id = jsonObject[XmlAttributeNames_1.XmlAttributeNames.Id];
        if (this.Owner != null && jsonObject[XmlAttributeNames_1.XmlAttributeNames.RootItemChangeKey]) {
            var rootItemChangeKey = jsonObject[XmlAttributeNames_1.XmlAttributeNames.RootItemChangeKey];
            if (!ExtensionMethods_1.StringHelper.IsNullOrEmpty(rootItemChangeKey)) {
                this.Owner.RootItemId.ChangeKey = rootItemChangeKey;
            }
        }
    }
    /**
     * @internal Loads from XMLjsObject.
     *
     * @param   {any}               jsonObject
     * @param   {ExchangeService}   service
     */
    LoadFromXmlJsObject(jsObject, service) {
        for (let key in jsObject) {
            switch (key) {
                case XmlElementNames_1.XmlElementNames.AttachmentId:
                    this.LoadAttachmentIdFromXMLJsObject(jsObject[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.Name:
                    this.name = jsObject[key];
                    break;
                case XmlElementNames_1.XmlElementNames.ContentType:
                    this.contentType = jsObject[key];
                    break;
                case XmlElementNames_1.XmlElementNames.ContentId:
                    this.contentId = jsObject[key];
                    break;
                case XmlElementNames_1.XmlElementNames.ContentLocation:
                    this.contentLocation = jsObject[key];
                    break;
                case XmlElementNames_1.XmlElementNames.Size:
                    this.size = ExtensionMethods_1.Convert.toInt(jsObject[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.LastModifiedTime:
                    this.lastModifiedTime = service.ConvertUniversalDateTimeStringToLocalDateTime(jsObject[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.IsInline:
                    this.isInline = ExtensionMethods_1.Convert.toBool(jsObject[key]);
                    break;
                default:
                    break;
            }
        }
    }
    /**
     * @internal Sets value of field.
     *
     * /remarks/    We override the base implementation. Attachments cannot be modified so any attempts the change a property on an existing attachment is an error.
     *
     * @param   {IRefParam<T>}      field   The field.
     * @param   {T}                 value   The value.
     */
    SetFieldValue(field, value) {
        this.ThrowIfThisIsNotNew();
        super.SetFieldValue(field, value);
    }
    /**
     * @internal Throws exception if this is not a new service object.
     */
    ThrowIfThisIsNotNew() {
        if (!this.IsNew) {
            throw new Error(Strings_1.Strings.AttachmentCannotBeUpdated); //InvalidOperationException
        }
    }
    //ReadElementsFromXmlJsObject(reader: any): boolean { throw new Error("Attachment.ts - TryReadElementFromXmlJsObject : Not implemented."); }
    /**
     * @internal Validates this instance.
     *
     * @param   {number}   attachmentIndex   Index of this attachment.
     */
    Validate(attachmentIndex) {
    }
    /**
     * @internal Writes elements to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.Name, this.Name);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.ContentType, this.ContentType);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.ContentId, this.ContentId);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.ContentLocation, this.ContentLocation);
        if (writer.Service.RequestedServerVersion > ExchangeVersion_1.ExchangeVersion.Exchange2007_SP1) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.IsInline, this.IsInline);
        }
    }
}
exports.Attachment = Attachment;
//# sourceMappingURL=Attachment.js.map