"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OofSettings = void 0;
const ArgumentException_1 = require("../../Exceptions/ArgumentException");
const EwsUtilities_1 = require("../../Core/EwsUtilities");
const OofExternalAudience_1 = require("../../Enumerations/OofExternalAudience");
const OofReply_1 = require("../../Misc/Availability/OofReply");
const OofState_1 = require("../../Enumerations/OofState");
const Strings_1 = require("../../Strings");
const TimeWindow_1 = require("../../Misc/Availability/TimeWindow");
const XmlElementNames_1 = require("../../Core/XmlElementNames");
const XmlNamespace_1 = require("../../Enumerations/XmlNamespace");
const ComplexProperty_1 = require("../ComplexProperty");
/**
 * Represents a user's Out of Office (OOF) settings.
 */
class OofSettings extends ComplexProperty_1.ComplexProperty {
    /**
     * Initializes a new instance of OofSettings.
     */
    constructor() {
        super();
        this.state = 0;
        this.externalAudience = 0;
        this.allowExternalOof = 0;
        this.duration = null;
        this.internalReply = null;
        this.externalReply = null;
    }
    /**
     * Gets or sets the user's OOF state.
     *
     * @value The user's OOF state.
     */
    get State() {
        return this.state;
    }
    set State(value) {
        this.state = value;
    }
    /**
     * Gets or sets a value indicating who should receive external OOF messages.
     */
    get ExternalAudience() {
        return this.externalAudience;
    }
    set ExternalAudience(value) {
        this.externalAudience = value;
    }
    /**
     * Gets or sets the duration of the OOF status when State is set to OofState.Scheduled.
     */
    get Duration() {
        return this.duration;
    }
    set Duration(value) {
        this.duration = value;
    }
    /**
     * Gets or sets the OOF response sent other users in the user's domain or trusted domain.
     */
    get InternalReply() {
        return this.internalReply;
    }
    set InternalReply(value) {
        this.internalReply = value;
    }
    /**
     * Gets or sets the OOF response sent to addresses outside the user's domain or trusted domain.
     */
    get ExternalReply() {
        return this.externalReply;
    }
    set ExternalReply(value) {
        this.externalReply = value;
    }
    /**
     * Gets a value indicating the authorized external OOF notifications.
     */
    get AllowExternalOof() {
        return this.allowExternalOof;
    }
    set AllowExternalOof(value) {
        this.allowExternalOof = value;
    }
    /**
     * @internal Loads from xmlJsObject.
     *
     * @param   {any}               jsObject   The json property.
     * @param   {ExchangeService}   service        [description]
     */
    LoadFromXmlJsObject(jsObject, service) {
        for (var key in jsObject) {
            switch (key) {
                case XmlElementNames_1.XmlElementNames.OofState:
                    this.state = OofState_1.OofState[jsObject[key]];
                    break;
                case XmlElementNames_1.XmlElementNames.ExternalAudience:
                    this.externalAudience = OofExternalAudience_1.OofExternalAudience[jsObject[key]];
                    break;
                case XmlElementNames_1.XmlElementNames.Duration:
                    this.duration = new TimeWindow_1.TimeWindow();
                    this.duration.LoadFromXmlJsObject(jsObject[key], service);
                    break;
                case XmlElementNames_1.XmlElementNames.InternalReply:
                    this.internalReply = new OofReply_1.OofReply();
                    this.internalReply.LoadFromXmlJsObject(jsObject[key], service);
                    break;
                case XmlElementNames_1.XmlElementNames.ExternalReply:
                    this.externalReply = new OofReply_1.OofReply();
                    this.externalReply.LoadFromXmlJsObject(jsObject[key], service);
                    break;
                default:
                    break;
            }
        }
    }
    /**
     * Serializes an OofReply. Emits an empty OofReply in case the one passed in is null.
     *
     * @param   {OofReply}              oofReply         The oof reply.
     * @param   {EwsServiceXmlWriter}   writer           The writer.
     * @param   {string}                xmlElementName   Name of the XML element.
     */
    SerializeOofReply(oofReply, writer, xmlElementName) {
        if (oofReply != null) {
            oofReply.WriteToXml(writer, xmlElementName);
        }
        else {
            OofReply_1.OofReply.WriteEmptyReplyToXml(writer, xmlElementName);
        }
    }
    /**
     * Validates this instance.
     */
    Validate() {
        if (this.State == OofState_1.OofState.Scheduled) {
            if (this.Duration == null) {
                throw new ArgumentException_1.ArgumentException(Strings_1.Strings.DurationMustBeSpecifiedWhenScheduled);
            }
            EwsUtilities_1.EwsUtilities.ValidateParam(this.Duration, "Duration");
        }
    }
    /**
     * @internal Writes elements to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        super.WriteElementsToXml(writer);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.OofState, OofState_1.OofState[this.State]);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.ExternalAudience, OofExternalAudience_1.OofExternalAudience[this.ExternalAudience]);
        if (this.Duration != null && this.State == OofState_1.OofState.Scheduled) {
            this.Duration.WriteToXml(writer, XmlElementNames_1.XmlElementNames.Duration);
        }
        this.SerializeOofReply(this.InternalReply, writer, XmlElementNames_1.XmlElementNames.InternalReply);
        this.SerializeOofReply(this.ExternalReply, writer, XmlElementNames_1.XmlElementNames.ExternalReply);
    }
}
exports.OofSettings = OofSettings;
//# sourceMappingURL=OofSettings.js.map