"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComplexPropertyCollection = void 0;
const ArgumentException_1 = require("../Exceptions/ArgumentException");
const ExtensionMethods_1 = require("../ExtensionMethods");
const EwsLogging_1 = require("../Core/EwsLogging");
const EwsServiceJsonReader_1 = require("../Core/EwsServiceJsonReader");
const ServiceLocalException_1 = require("../Exceptions/ServiceLocalException");
const Strings_1 = require("../Strings");
const XmlNamespace_1 = require("../Enumerations/XmlNamespace");
const ComplexProperty_1 = require("./ComplexProperty");
/**
 * Represents a collection of properties that can be sent to and retrieved from EWS.
 *
 * @type <TComplexProperty>   ComplexProperty type.
 */
class ComplexPropertyCollection extends ComplexProperty_1.ComplexProperty {
    /**
     * @internal Initializes a new instance of the **ComplexPropertyCollection** class.
     *
     */
    constructor() {
        super();
        this.___typeGenerics = ["ComplexProperty"];
        this.items = [];
        this.addedItems = [];
        this.modifiedItems = [];
        this.removedItems = [];
    }
    /**
     * @internal Gets the items. (workaround for GetEnumerator)
     *
     * @return The items.
     */
    get Items() { return this.items; }
    /**
     * @internal Gets the added items.
     *
     * @return The added items.
     */
    get AddedItems() { return this.addedItems; }
    /**
     * @internal Gets the modified items.
     *
     * @return The modified items
     */
    get ModifiedItems() { return this.modifiedItems; }
    /**
     * @internal Gets the removed items.
     *
     * @return The removed items.
     */
    get RemovedItems() { return this.removedItems; }
    /**
     * Gets the total number of properties in the collection.
     */
    get Count() { return this.items.length; }
    /**
     * Gets the property at the specified index.
     *
     * @param   {number}   index   The zero-based index of the property to get.
     * @return  {TComplexProperty}           The property at the specified index.
     */
    _getItem(index) {
        if (index < 0 || index >= this.Count) {
            throw new ArgumentException_1.ArgumentOutOfRangeException("index", Strings_1.Strings.IndexIsOutOfRange);
        }
        return this.items[index];
    }
    /**
     * @internal Clears the change log.
     */
    ClearChangeLog() {
        this.removedItems.splice(0);
        this.addedItems.splice(0);
        this.modifiedItems.splice(0);
    }
    /**
     * Determines whether a specific property is in the collection.
     *
     * @param   {TComplexProperty}   complexProperty   The property to locate in the collection.
     * @return  {boolean}           True if the property was found in the collection, false otherwise.
     */
    Contains(complexProperty) { return this.items.indexOf(complexProperty) >= 0; }
    /**
     * @internal Loads from XMLJsObject collection to create a new collection item.
     *
     * @interface   IJsonCollectionDeserializer
     *
     * @param   {any}               jsObjectCollection   The json collection.
     * @param   {ExchangeService}   service          The service.
     */
    CreateFromXmlJsObjectCollection(jsObjectCollection, service) {
        let collection = jsObjectCollection;
        if (!ExtensionMethods_1.ArrayHelper.isArray(collection)) {
            collection = [];
            let collectionElement = this.GetCollectionItemXmlElementName(null);
            let typeName = ExtensionMethods_1.TypeSystem.GetJsObjectTypeName(jsObjectCollection);
            if (collectionElement === null || (typeName && collectionElement === typeName)) {
                collection = [jsObjectCollection];
            }
            else {
                for (let key in jsObjectCollection) {
                    if (key.indexOf("__") === 0) //skip xmljsobject conversion entries like __type and __prefix
                        continue;
                    let collectionObj = jsObjectCollection[key];
                    if (!ExtensionMethods_1.ArrayHelper.isArray(collectionObj)) {
                        collectionObj = EwsServiceJsonReader_1.EwsServiceJsonReader.ReadAsArray(jsObjectCollection, key);
                    }
                    ExtensionMethods_1.ArrayHelper.AddRange(collection, collectionObj);
                }
            }
        }
        for (let jsonObject of collection) {
            let jsonProperty = jsonObject;
            if (jsonProperty != null) {
                let complexProperty = null;
                // If type property is present, use it. Otherwise create default property instance.
                // Note: polymorphic collections (such as Attachments) need a type property so
                // the CreateDefaultComplexProperty call will fail.
                if (ExtensionMethods_1.TypeSystem.GetJsObjectTypeName(jsonProperty)) {
                    complexProperty = this.CreateComplexProperty(ExtensionMethods_1.TypeSystem.GetJsObjectTypeName(jsonProperty));
                }
                else {
                    complexProperty = this.CreateDefaultComplexProperty();
                }
                if (complexProperty != null) {
                    complexProperty.LoadFromXmlJsObject(jsonProperty, service);
                    this.InternalAdd(complexProperty, true);
                }
            }
        }
    }
    /**
     *  Returns an enumerator that iterates through the collection. this case this.items
     */
    GetEnumerator() {
        return this.items;
    }
    /**
     * Searches for a specific property and return its zero-based index within the collection.
     *
     * @param   {TComplexProperty}   complexProperty   The property to locate in the collection.
     * @return  {number}                     The zero-based index of the property within the collection.
     */
    IndexOf(complexProperty) {
        return this.items.indexOf(complexProperty);
    }
    InternalAdd(complexProperty, loading = false) {
        EwsLogging_1.EwsLogging.Assert(complexProperty != null, "ComplexPropertyCollection.InternalAdd", "complexProperty is null");
        if (this.items.indexOf(complexProperty) < 0) {
            this.items.push(complexProperty);
            if (!loading) {
                ExtensionMethods_1.ArrayHelper.RemoveEntry(this.removedItems, complexProperty); // this.removedItems.Remove(complexProperty);
                this.addedItems.push(complexProperty);
            }
            complexProperty.OnChange.push(this.ItemChanged.bind(this));
            this.Changed();
        }
    }
    /**
     * @internal Clear collection.
     */
    InternalClear() {
        while (this.Count > 0) {
            this.InternalRemoveAt(0);
        }
    }
    /**
     * @internal Remove specified complex property.
     *
     * @param   {TComplexProperty}   complexProperty   The complex property.
     * @return  {boolean}           True if the complex property was successfully removed from the collection, false otherwise.
     */
    InternalRemove(complexProperty) {
        EwsLogging_1.EwsLogging.Assert(complexProperty != null, "ComplexPropertyCollection.InternalRemove", "complexProperty is null");
        if (ExtensionMethods_1.ArrayHelper.RemoveEntry(this.items, complexProperty)) { // this.items.Remove(complexProperty))
            ExtensionMethods_1.ArrayHelper.RemoveEntry(complexProperty.OnChange, this.ItemChanged); // complexProperty.OnChange -= this.ItemChanged;
            if (this.addedItems.indexOf(complexProperty) < 0) {
                this.removedItems.push(complexProperty);
            }
            else {
                ExtensionMethods_1.ArrayHelper.RemoveEntry(this.addedItems, complexProperty); // this.addedItems.Remove(complexProperty);
            }
            ExtensionMethods_1.ArrayHelper.RemoveEntry(this.modifiedItems, complexProperty); // this.modifiedItems.Remove(complexProperty);
            this.Changed();
            return true;
        }
        else {
            return false;
        }
    }
    /**
     * @internal Remote entry at index.
     *
     * @param   {number}   index   The index.
     */
    InternalRemoveAt(index) {
        EwsLogging_1.EwsLogging.Assert(index >= 0 && index < this.Count, "ComplexPropertyCollection.InternalRemoveAt", "index is out of range.");
        this.InternalRemove(this.items[index]);
    }
    /**
     * @internal Item changed.
     *
     * @param   {ComplexProperty}   complexProperty   The complex property.
     */
    ItemChanged(complexProperty) {
        //TComplexProperty property = complexProperty as TComplexProperty;
        let property = complexProperty;
        // EwsLogging.Assert(
        //     property != null,
        //     "ComplexPropertyCollection.ItemChanged",
        //     StringHelper.Format("ComplexPropertyCollection.ItemChanged: the type of the complexProperty argument ({0}) is not supported.", complexProperty.___typeName));
        if (this.addedItems.indexOf(property) < 0) {
            if (this.modifiedItems.indexOf(property) < 0) {
                this.modifiedItems.push(property);
                this.Changed();
            }
        }
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}				jsObject	Json Object converted from XML.
     * @param   {ExchangeService}	service	The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        EwsLogging_1.EwsLogging.Assert(false, "ComplexPropertyCollection.LoadFromXmlJsObject", "LoadFromXmlJsObject was called, should not be calling. Fix it to direct to Create or Update call instad.");
        this.CreateFromXmlJsObjectCollection(jsObject, service);
    }
    /**
     * @internal Removes from change log.
     *
     * @param   {TComplexProperty}   complexProperty   The complex property.
     */
    RemoveFromChangeLog(complexProperty) {
        ExtensionMethods_1.ArrayHelper.RemoveEntry(this.removedItems, complexProperty); //this.removedItems.Remove(complexProperty);
        ExtensionMethods_1.ArrayHelper.RemoveEntry(this.modifiedItems, complexProperty); //this.modifiedItems.Remove(complexProperty);
        ExtensionMethods_1.ArrayHelper.RemoveEntry(this.addedItems, complexProperty); //this.addedItems.Remove(complexProperty);
    }
    /**
     * @internal Determine whether we should write collection to XML or not.
     *
     * @return  {boolean}      True if collection contains at least one element.
     */
    ShouldWriteToRequest() {
        // Only write collection if it has at least one element.
        return this.Count > 0;
    }
    /**
     * @internal Loads from XMLJsObject collection to update collection Items.
     *
     * @interface   IJsonCollectionDeserializer
     *
     * @param   {any}               jsObjectCollection   The XMLJsObject collection.
     * @param   {ExchangeService}   service          The service.
     */
    UpdateFromXmlJsObjectCollection(jsObjectCollection, service) {
        let collection = jsObjectCollection;
        if (!ExtensionMethods_1.ArrayHelper.isArray(collection)) {
            collection = [];
            let collectionElement = this.GetCollectionItemXmlElementName(null);
            let typeName = ExtensionMethods_1.TypeSystem.GetJsObjectTypeName(jsObjectCollection);
            if (collectionElement === null || (typeName && collectionElement === typeName)) {
                collection = [jsObjectCollection];
            }
            else {
                for (let key in jsObjectCollection) {
                    if (key.indexOf("__") === 0) //skip xmljsobject conversion entries like __type and __prefix
                        continue;
                    let collectionObj = jsObjectCollection[key];
                    if (!ExtensionMethods_1.ArrayHelper.isArray(collectionObj)) {
                        collectionObj = EwsServiceJsonReader_1.EwsServiceJsonReader.ReadAsArray(jsObjectCollection, key);
                    }
                    ExtensionMethods_1.ArrayHelper.AddRange(collection, collectionObj);
                }
            }
        }
        if (this.Count != collection.length) {
            throw new ServiceLocalException_1.ServiceLocalException(Strings_1.Strings.PropertyCollectionSizeMismatch);
        }
        let index = 0;
        for (let jsonObject of collection) {
            let jsonProperty = jsonObject;
            if (jsonProperty != null) {
                let expectedComplexProperty = null;
                if (ExtensionMethods_1.TypeSystem.GetJsObjectTypeName(jsonProperty)) {
                    expectedComplexProperty = this.CreateComplexProperty(ExtensionMethods_1.TypeSystem.GetJsObjectTypeName(jsonProperty));
                }
                else {
                    expectedComplexProperty = this.CreateDefaultComplexProperty();
                }
                let actualComplexProperty = this._getItem(index++);
                if (expectedComplexProperty == null || !(actualComplexProperty instanceof expectedComplexProperty.constructor)) {
                    throw new ServiceLocalException_1.ServiceLocalException(Strings_1.Strings.PropertyTypeIncompatibleWhenUpdatingCollection);
                }
                actualComplexProperty.LoadFromXmlJsObject(jsonProperty, service);
            }
            else {
                throw new ServiceLocalException_1.ServiceLocalException();
            }
        }
    }
    /**
     * @internal Writes elements to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        for (let complexProperty of this.items) {
            complexProperty.WriteToXml(writer, this.GetCollectionItemXmlElementName(complexProperty));
        }
    }
    /**
     * @internal Writes to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer           The writer.
     * @param   {XmlNamespace}          xmlNamespace     The XML namespace.
     * @param   {string}                xmlElementName   Name of the XML element.
     */
    WriteToXml(writer, xmlElementName, xmlNamespace = XmlNamespace_1.XmlNamespace.Types) {
        if (this.ShouldWriteToRequest()) {
            super.WriteToXml(writer, xmlElementName, xmlNamespace);
        }
    }
    /**
     * @internal Writes the update to XML.
     * ICustomUpdateSerializer.WriteSetUpdateToXml
     *
     * @param   {EwsServiceXmlWriter}   writer               The writer.
     * @param   {ServiceObject}         ewsObject            The ews object.
     * @param   {PropertyDefinition}    propertyDefinition   Property definition.
     * @return  {boolean}               True if property generated serialization.
     */
    WriteSetUpdateToXml(writer, ewsObject, propertyDefinition) {
        // If the collection is empty, delete the property.
        if (this.Count == 0) {
            writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, ewsObject.GetDeleteFieldXmlElementName());
            propertyDefinition.WriteToXml(writer);
            writer.WriteEndElement();
            return true;
        }
        // Otherwise, use the default XML serializer.
        else {
            return false;
        }
    }
    /**
     * @internal Writes the deletion update to XML.
     * ICustomUpdateSerializer.WriteDeleteUpdateToXml
     *
     * @param   {EwsServiceXmlWriter}   writer      The writer.
     * @param   {ServiceObject}         ewsObject   The ews object.
     * @return  {boolean}               True if property generated serialization.
     */
    WriteDeleteUpdateToXml(writer, ewsObject) {
        // Use the default XML serializer.
        return false;
    }
}
exports.ComplexPropertyCollection = ComplexPropertyCollection;
//# sourceMappingURL=ComplexPropertyCollection.js.map