"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GroupMemberCollection = void 0;
const ArgumentException_1 = require("../Exceptions/ArgumentException");
const ComplexPropertyCollection_1 = require("./ComplexPropertyCollection");
const ContactGroupSchema_1 = require("../Core/ServiceObjects/Schemas/ContactGroupSchema");
const EmailAddress_1 = require("./EmailAddress");
const EwsLogging_1 = require("../Core/EwsLogging");
const EwsUtilities_1 = require("../Core/EwsUtilities");
const GroupMember_1 = require("./GroupMember");
const GroupMemberPropertyDefinition_1 = require("../PropertyDefinitions/GroupMemberPropertyDefinition");
const MailboxType_1 = require("../Enumerations/MailboxType");
const ServiceValidationException_1 = require("../Exceptions/ServiceValidationException");
const ExtensionMethods_1 = require("../ExtensionMethods");
const Strings_1 = require("../Strings");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const XmlNamespace_1 = require("../Enumerations/XmlNamespace");
/**
 * Represents a collection of members of GroupMember type.
 * @sealed
 */
class GroupMemberCollection extends ComplexPropertyCollection_1.ComplexPropertyCollection {
    /**
     * Initializes a new instance of the  class.
     *
     */
    constructor() {
        super();
        /**
         * If the collection is cleared, then store PDL members collection is updated with "SetItemField". If the collection is not cleared, then store PDL members collection is updated with "AppendToItemField".
         *
         */
        this.collectionIsCleared = false;
    }
    /**
     * Adds a member to the collection.
     *
     * @param   {GroupMember}   member   The member to add.
     */
    Add(member) {
        EwsUtilities_1.EwsUtilities.ValidateParam(member, "member");
        EwsLogging_1.EwsLogging.Assert(member.Key == null, "GroupMemberCollection.Add", "member.Key is not null.");
        EwsLogging_1.EwsLogging.Assert(!this.Contains(member), "GroupMemberCollection.Add", "The member is already in the collection");
        this.InternalAdd(member);
    }
    /**
     * Adds a member that is linked to a specific e-mail address of a contact.
     *
     * @param   {Contact}   contact           The contact to link to.
     * @param   {EmailAddressKey}   emailAddressKey   The contact's e-mail address to link to.
     */
    AddContactEmailAddress(contact, emailAddressKey) {
        this.Add(new GroupMember_1.GroupMember(contact, emailAddressKey));
    }
    /**
     * Adds a member linked to a Contact Group.
     *
     * @param   {ItemId}   contactGroupId   The Id of the contact group.
     */
    AddContactGroup(contactGroupId) {
        this.Add(new GroupMember_1.GroupMember(contactGroupId));
    }
    AddDirectoryContact(address, routingType = EmailAddress_1.EmailAddress.SmtpRoutingType) {
        this.Add(new GroupMember_1.GroupMember(address, routingType, MailboxType_1.MailboxType.Contact));
    }
    /**
     * Adds a member linked to a mail-enabled Public Folder.
     *
     * @param   {string}   smtpAddress   The SMTP address of the mail-enabled Public Folder.
     */
    AddDirectoryPublicFolder(smtpAddress) {
        this.Add(new GroupMember_1.GroupMember(smtpAddress, EmailAddress_1.EmailAddress.SmtpRoutingType, MailboxType_1.MailboxType.PublicFolder));
    }
    AddDirectoryUser(address, routingType = EmailAddress_1.EmailAddress.SmtpRoutingType) {
        this.Add(new GroupMember_1.GroupMember(address, routingType, MailboxType_1.MailboxType.Mailbox));
    }
    AddOneOff(displayName, address, routingType = EmailAddress_1.EmailAddress.SmtpRoutingType) {
        this.Add(new GroupMember_1.GroupMember(displayName, address, routingType));
    }
    AddPersonalContact(contactId, addressToLink = null) {
        this.Add(new GroupMember_1.GroupMember(contactId, addressToLink));
    }
    /**
     * Adds a member linked to a Public Group.
     *
     * @param   {string}   smtpAddress   The SMTP address of the Public Group.
     */
    AddPublicGroup(smtpAddress) {
        this.Add(new GroupMember_1.GroupMember(smtpAddress, EmailAddress_1.EmailAddress.SmtpRoutingType, MailboxType_1.MailboxType.PublicGroup));
    }
    /**
     * Adds multiple members to the collection.
     *
     * @param   {GroupMember[]}   members   The members to add.
     */
    AddRange(members /*IEnumerable<T>*/) {
        EwsUtilities_1.EwsUtilities.ValidateParam(members, "members");
        for (let member of members) {
            this.Add(member);
        }
    }
    /**
     * Clears the collection.
     */
    Clear() {
        // mark the whole collection for deletion
        this.InternalClear();
        this.collectionIsCleared = true;
    }
    /**
     * @internal Clears the change log.
     *
     */
    ClearChangeLog() {
        super.ClearChangeLog();
        this.collectionIsCleared = false;
    }
    /**
     * @internal Creates a GroupMember object from an XML element name.
     *
     * @param   {string}        xmlElementName   The XML element name from which to create the e-mail address.
     * @return  {GroupMember}   An GroupMember object.
     */
    CreateComplexProperty(xmlElementName) {
        return new GroupMember_1.GroupMember();
    }
    /**
     * @internal Creates the default complex property.
     *
     * @return  {GroupMember}      An GroupMember object.
     */
    CreateDefaultComplexProperty() {
        return new GroupMember_1.GroupMember();
    }
    /**
     * Finds the member with the specified key in the collection.
     * Members that have not yet been saved do not have a key.
     *
     * @param   {}   key   The key of the member to find.
     * @return  {}         The member with the specified key.
     */
    Find(key) {
        EwsUtilities_1.EwsUtilities.ValidateParam(key, "key");
        for (let item of this.Items) {
            if (item.Key == key) {
                return item;
            }
        }
        return null;
    }
    /**
     * @internal Retrieves the XML element name corresponding to the provided GroupMember object.
     *
     * @param   {GroupMember}   member   The GroupMember object from which to determine the XML element name.
     * @return  {string}            The XML element name corresponding to the provided GroupMember object.
     */
    GetCollectionItemXmlElementName(member) {
        return XmlElementNames_1.XmlElementNames.Member;
    }
    /**
     * @internal Validates this instance.
     */
    InternalValidate() {
        super.InternalValidate();
        for (let groupMember of this.ModifiedItems) {
            if (ExtensionMethods_1.StringHelper.IsNullOrEmpty(groupMember.Key)) {
                throw new ServiceValidationException_1.ServiceValidationException(Strings_1.Strings.ContactGroupMemberCannotBeUpdatedWithoutBeingLoadedFirst);
            }
        }
    }
    /**
     * Removes a member from the collection.
     *
     * @param   {GroupMember}   member   The member to remove.
     * @return  {boolean}       True if the group member was successfully removed from the collection, false otherwise.
     */
    Remove(member) {
        return this.InternalRemove(member);
    }
    /**
     * Removes a member at the specified index.
     *
     * @param   {number}   index   The index of the member to remove.
     */
    RemoveAt(index) {
        if (index < 0 || index >= this.Count) {
            throw new ArgumentException_1.ArgumentOutOfRangeException("index", Strings_1.Strings.IndexIsOutOfRange);
        }
        this.InternalRemoveAt(index);
    }
    /**
     * Delete the whole members collection.
     *
     * @param   {EwsServiceXmlWriter}   writer   Xml writer.
     */
    WriteDeleteMembersCollectionToXml(writer) {
        writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.DeleteItemField);
        ContactGroupSchema_1.ContactGroupSchema.Members.WriteToXml(writer);
        writer.WriteEndElement();
    }
    /**
     * Generate XML to delete individual members.
     *
     * @param   {EwsServiceXmlWriter}   writer    Xml writer.
     * @param   {GroupMember[]}         members   Members to delete.
     */
    WriteDeleteMembersToXml(writer, members /* List<GroupMember>*/) {
        if (members.length != 0) {
            let memberPropDef = new GroupMemberPropertyDefinition_1.GroupMemberPropertyDefinition();
            for (let member of members) {
                writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.DeleteItemField);
                memberPropDef.Key = member.Key;
                memberPropDef.WriteToXml(writer);
                writer.WriteEndElement(); // DeleteItemField
            }
        }
    }
    /**
     * Generate XML to Set or Append members. When members are set, the existing PDL member collection is cleared On append members are added to the PDL existing members collection.
     *
     * @param   {EwsServiceXmlWriter}   writer    Xml writer.
     * @param   {GroupMember[]}         members   Members to set or append.
     * @param   {boolean}               setMode   True - set members, false - append members.
     */
    WriteSetOrAppendMembersToXml(writer, members /*List<GroupMember>*/, setMode) {
        if (members.length != 0) {
            writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, setMode ? XmlElementNames_1.XmlElementNames.SetItemField : XmlElementNames_1.XmlElementNames.AppendToItemField);
            ContactGroupSchema_1.ContactGroupSchema.Members.WriteToXml(writer);
            writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.DistributionList);
            writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.Members);
            for (let member of members) {
                member.WriteToXml(writer, XmlElementNames_1.XmlElementNames.Member);
            }
            writer.WriteEndElement(); // Members
            writer.WriteEndElement(); // Group
            writer.WriteEndElement(); // setMode ? SetItemField : AppendItemField
        }
    }
    //#region ICustomUpdateSerializer
    /**
     * @internal Writes the deletion update to XML.
     * ICustomUpdateSerializer.WriteDeleteUpdateToXml
     *
     * @param   {EwsServiceXmlWriter}   writer      The writer.
     * @param   {ServiceObject}         ewsObject   The ews object.
     * @return  {boolean}               True if property generated serialization.
     */
    WriteDeleteUpdateToXml(writer, ewsObject) {
        return false;
    }
    /**
     * @internal Writes the update to XML.
     * ICustomUpdateSerializer.WriteSetUpdateToXml
     *
     * @param   {EwsServiceXmlWriter}   writer               The writer.
     * @param   {ServiceObject}         ewsObject            The ews object.
     * @param   {PropertyDefinition}    propertyDefinition   Property definition.
     * @return  {boolean}               True if property generated serialization.
     */
    WriteSetUpdateToXml(writer, ewsObject, propertyDefinition) {
        if (this.collectionIsCleared) {
            if (this.AddedItems.length == 0) {
                // Delete the whole members collection
                this.WriteDeleteMembersCollectionToXml(writer);
            }
            else {
                // The collection is cleared, so Set
                this.WriteSetOrAppendMembersToXml(writer, this.AddedItems, true);
            }
        }
        else {
            // The collection is not cleared, i.e. dl.Members.Clear() is not called.
            // Append AddedItems.
            this.WriteSetOrAppendMembersToXml(writer, this.AddedItems, false);
            // Since member replacement is not supported by server
            // Delete old ModifiedItems, then recreate new instead.
            this.WriteDeleteMembersToXml(writer, this.ModifiedItems);
            this.WriteSetOrAppendMembersToXml(writer, this.ModifiedItems, false);
            // Delete RemovedItems.
            this.WriteDeleteMembersToXml(writer, this.RemovedItems);
        }
        return true;
    }
}
exports.GroupMemberCollection = GroupMemberCollection;
//# sourceMappingURL=GroupMemberCollection.js.map