"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MeetingTimeZone = void 0;
const EwsUtilities_1 = require("../Core/EwsUtilities");
const ExtensionMethods_1 = require("../ExtensionMethods");
const TimeChange_1 = require("./TimeChange");
const TimeZoneInfo_1 = require("../TimeZoneInfo");
const XmlAttributeNames_1 = require("../Core/XmlAttributeNames");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const XmlNamespace_1 = require("../Enumerations/XmlNamespace");
const ComplexProperty_1 = require("./ComplexProperty");
/**
 * @internal Represents a time zone in which a meeting is defined.
 * @sealed
 */
class MeetingTimeZone extends ComplexProperty_1.ComplexProperty {
    constructor(timeZoneOrName = null) {
        super();
        this.name = null;
        this.baseOffset = null;
        this.standard = null;
        this.daylight = null;
        if (arguments.length === 1) {
            if (timeZoneOrName instanceof TimeZoneInfo_1.TimeZoneInfo) {
                // Unfortunately, MeetingTimeZone does not support all the time transition types
                // supported by TimeZoneInfo. That leaves us unable to accurately convert TimeZoneInfo
                // into MeetingTimeZone. So we don't... Instead, we emit the time zone's Id and
                // hope the server will find a match (which it should).
                this.Name = timeZoneOrName.Id;
            }
            else {
                this.name = timeZoneOrName;
            }
        }
    }
    /**
     * Gets or sets the name of the time zone.
     */
    get Name() {
        return this.name;
    }
    set Name(value) {
        this.SetFieldValue({ getValue: () => this.name, setValue: (fieldValue) => { this.name = fieldValue; } }, value);
    }
    /**
     * Gets or sets the base offset of the time zone from the UTC time zone.
     */
    get BaseOffset() {
        return this.baseOffset;
    }
    set BaseOffset(value) {
        this.SetFieldValue({ getValue: () => this.baseOffset, setValue: (fieldValue) => { this.baseOffset = fieldValue; } }, value);
    }
    /**
     * Gets or sets a TimeChange defining when the time changes to Standard Time.
     */
    get Standard() {
        return this.standard;
    }
    set Standard(value) {
        this.SetFieldValue({ getValue: () => this.standard, setValue: (fieldValue) => { this.standard = fieldValue; } }, value);
    }
    /**
     * Gets or sets a TimeChange defining when the time changes to Daylight Saving Time.
     */
    get Daylight() {
        return this.daylight;
    }
    set Daylight(value) {
        this.SetFieldValue({ getValue: () => this.daylight, setValue: (fieldValue) => { this.daylight = fieldValue; } }, value);
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}                 jsObject                Json Object converted from XML.
     * @param   {ExchangeService}     service                 The service.
     */
    LoadFromXmlJsObject(jsonProperty, service) {
        for (let key in jsonProperty) {
            switch (key) {
                case XmlElementNames_1.XmlElementNames.BaseOffset:
                    this.baseOffset = EwsUtilities_1.EwsUtilities.XSDurationToTimeSpan(jsonProperty[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.Standard:
                    this.standard = new TimeChange_1.TimeChange();
                    this.standard.LoadFromXmlJsObject(jsonProperty[key], service);
                    break;
                case XmlElementNames_1.XmlElementNames.Daylight:
                    this.daylight = new TimeChange_1.TimeChange();
                    this.daylight.LoadFromXmlJsObject(jsonProperty[key], service);
                    break;
                case XmlAttributeNames_1.XmlAttributeNames.TimeZoneName:
                    this.name = jsonProperty[key];
                    break;
                default:
                    break;
            }
        }
    }
    /**
     * @internal Converts this meeting time zone into a TimeZoneInfo structure.
    */
    ToTimeZoneInfo() {
        // MeetingTimeZone.ToTimeZoneInfo throws ArgumentNullException if name is null
        // TimeZoneName is optional, may not show in the response.
        if (ExtensionMethods_1.StringHelper.IsNullOrEmpty(this.Name)) {
            return null;
        }
        let result = null;
        try {
            result = TimeZoneInfo_1.TimeZoneInfo.FindSystemTimeZoneById(this.Name);
        }
        catch (exception) {
            // Could not find a time zone with that Id on the local system.
        }
        // Again, we cannot accurately convert MeetingTimeZone into TimeZoneInfo
        // because TimeZoneInfo doesn't support absolute date transitions. So if
        // there is no system time zone that has a matching Id, we return null.
        return result;
    }
    /**
     * @internal Writes the attributes to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteAttributesToXml(writer) {
        writer.WriteAttributeValue(XmlAttributeNames_1.XmlAttributeNames.TimeZoneName, this.Name);
    }
    /**
     * @internal Writes elements to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        if (this.BaseOffset) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.BaseOffset, EwsUtilities_1.EwsUtilities.TimeSpanToXSDuration(this.BaseOffset));
        }
        if (this.Standard != null) {
            this.Standard.WriteToXml(writer, XmlElementNames_1.XmlElementNames.Standard);
        }
        if (this.Daylight != null) {
            this.Daylight.WriteToXml(writer, XmlElementNames_1.XmlElementNames.Daylight);
        }
    }
}
exports.MeetingTimeZone = MeetingTimeZone;
//# sourceMappingURL=MeetingTimeZone.js.map