"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NumberedRecurrenceRange = void 0;
const ExtensionMethods_1 = require("../../../ExtensionMethods");
const XmlElementNames_1 = require("../../../Core/XmlElementNames");
const XmlNamespace_1 = require("../../../Enumerations/XmlNamespace");
const RecurrenceRange_1 = require("./RecurrenceRange");
/**
 * @internal Represents recurrence range with start and number of occurance.
 */
class NumberedRecurrenceRange extends RecurrenceRange_1.RecurrenceRange {
    constructor(startDate = null, numberOfOccurrences = null) {
        if (arguments.length === 0) {
            super();
            return;
        }
        super(startDate);
        this.numberOfOccurrences = numberOfOccurrences;
    }
    /**
     * @internal Gets the name of the XML element.
     *
     * @value   The name of the XML element.
     */
    get XmlElementName() {
        return XmlElementNames_1.XmlElementNames.NumberedRecurrence;
    }
    /**
     * Gets or sets the number of occurrences.
     *
     * @value   The number of occurrences.
     */
    get NumberOfOccurrences() {
        return this.numberOfOccurrences;
    }
    set NumberOfOccurrences(value) {
        this.SetFieldValue({ getValue: () => this.numberOfOccurrences, setValue: (updateValue) => { this.numberOfOccurrences = updateValue; } }, value);
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}                 jsObject                Json Object converted from XML.
     * @param   {ExchangeService}     service                 The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        super.LoadFromXmlJsObject(jsObject, service);
        for (let key in jsObject) {
            switch (key) {
                case XmlElementNames_1.XmlElementNames.NumberOfOccurrences:
                    this.numberOfOccurrences = ExtensionMethods_1.Convert.toNumber(jsObject[key]);
                    break;
                default:
                    break;
            }
        }
    }
    /**
     * Setups the recurrence.
     *
     * @param   {Recurrence}   recurrence   The recurrence.
     */
    SetupRecurrence(recurrence) {
        super.SetupRecurrence(recurrence);
        recurrence.NumberOfOccurrences = this.NumberOfOccurrences;
    }
    /**
     * @internal Writes elements to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        super.WriteElementsToXml(writer);
        if (this.NumberOfOccurrences) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.NumberOfOccurrences, this.NumberOfOccurrences);
        }
    }
}
exports.NumberedRecurrenceRange = NumberedRecurrenceRange;
//# sourceMappingURL=NumberedRecurrenceRange.js.map