"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rule = void 0;
const EwsUtilities_1 = require("../Core/EwsUtilities");
const RuleActions_1 = require("./RuleActions");
const RulePredicates_1 = require("./RulePredicates");
const ExtensionMethods_1 = require("../ExtensionMethods");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const XmlNamespace_1 = require("../Enumerations/XmlNamespace");
const ComplexProperty_1 = require("./ComplexProperty");
/**
 * Represents a rule that automatically handles incoming messages.
 * A rule consists of a set of conditions and exceptions that determine whether or not a set of actions should be executed on incoming messages.
 *
 * @sealed
 */
class Rule extends ComplexProperty_1.ComplexProperty {
    /**
     * Initializes a new instance of the **Rule** class.
     */
    constructor() {
        super();
        /**
         * The rule ID.
         */
        this.ruleId = null;
        /**
         * The rule display name.
         */
        this.displayName = null;
        /**
         * The rule priority.
         */
        this.priority = 0;
        /**
         * The rule status of enabled or not.
         */
        this.isEnabled = false;
        /**
         * The rule status of is supported or not.
         */
        this.isNotSupported = false;
        /**
         * The rule status of in error or not.
         */
        this.isInError = false;
        /**
         * The rule conditions.
         */
        this.conditions = null;
        /**
         * The rule actions.
         */
        this.actions = null;
        /**
         * The rule exceptions.
         */
        this.exceptions = null;
        //// New rule has priority as 0 by default //todo: fix 0 or 1 when it it changes in offical repo
        this.priority = 1;
        //// New rule is enabled by default
        this.isEnabled = true;
        this.conditions = new RulePredicates_1.RulePredicates();
        this.actions = new RuleActions_1.RuleActions();
        this.exceptions = new RulePredicates_1.RulePredicates();
    }
    /**
     * Gets or sets the Id of this rule.
     */
    get Id() {
        return this.ruleId;
    }
    set Id(value) {
        this.SetFieldValue({ getValue: () => this.Id, setValue: (updateValue) => { this.Id = updateValue; } }, value);
    }
    /**
     * Gets or sets the name of this rule as it should be displayed to the user.
     */
    get DisplayName() {
        return this.displayName;
    }
    set DisplayName(value) {
        this.SetFieldValue({ getValue: () => this.displayName, setValue: (updateValue) => { this.displayName = updateValue; } }, value);
    }
    /**
     * Gets or sets the priority of this rule, which determines its execution order.
     */
    get Priority() {
        return this.priority;
    }
    set Priority(value) {
        this.SetFieldValue({ getValue: () => this.priority, setValue: (updateValue) => { this.priority = updateValue; } }, value);
    }
    /**
     * Gets or sets a value indicating whether this rule is enabled.
     */
    get IsEnabled() {
        return this.isEnabled;
    }
    set IsEnabled(value) {
        this.SetFieldValue({ getValue: () => this.isEnabled, setValue: (updateValue) => { this.isEnabled = updateValue; } }, value);
    }
    /**
     * Gets a value indicating whether this rule can be modified via EWS. If IsNotSupported is true, the rule cannot be modified via EWS.
     */
    get IsNotSupported() {
        return this.isNotSupported;
    }
    /**
     * Gets or sets a value indicating whether this rule has errors. A rule that is in error cannot be processed unless it is updated and the error is corrected.
     */
    get IsInError() {
        return this.isInError;
    }
    set IsInError(value) {
        this.SetFieldValue({ getValue: () => this.isInError, setValue: (updateValue) => { this.isInError = updateValue; } }, value);
    }
    /**
     * Gets the conditions that determine whether or not this rule should be executed against incoming messages.
     */
    get Conditions() {
        return this.conditions;
    }
    /**
     * Gets the actions that should be executed against incoming messages if the conditions evaluate as true.
     */
    get Actions() {
        return this.actions;
    }
    /**
     * Gets the exceptions that determine if this rule should be skipped even if its conditions evaluate to true.
     */
    get Exceptions() {
        return this.exceptions;
    }
    /**
     * @internal Validates this instance.
     */
    InternalValidate() {
        super.InternalValidate();
        EwsUtilities_1.EwsUtilities.ValidateParam(this.displayName, "DisplayName");
        EwsUtilities_1.EwsUtilities.ValidateParam(this.conditions, "Conditions");
        EwsUtilities_1.EwsUtilities.ValidateParam(this.exceptions, "Exceptions");
        EwsUtilities_1.EwsUtilities.ValidateParam(this.actions, "Actions");
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}                 jsObject                Json Object converted from XML.
     * @param   {ExchangeService}     service                 The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        for (let key in jsObject) {
            switch (key) {
                case XmlElementNames_1.XmlElementNames.DisplayName:
                    this.displayName = jsObject[key];
                    break;
                case XmlElementNames_1.XmlElementNames.RuleId:
                    this.ruleId = jsObject[key];
                    break;
                case XmlElementNames_1.XmlElementNames.Priority:
                    this.priority = ExtensionMethods_1.Convert.toNumber(jsObject[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.IsEnabled:
                    this.isEnabled = ExtensionMethods_1.Convert.toBool(jsObject[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.IsNotSupported:
                    this.isNotSupported = ExtensionMethods_1.Convert.toBool(jsObject[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.IsInError:
                    this.isInError = ExtensionMethods_1.Convert.toBool(jsObject[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.Conditions:
                    this.conditions.LoadFromXmlJsObject(jsObject[key], service);
                    break;
                case XmlElementNames_1.XmlElementNames.Actions:
                    this.actions.LoadFromXmlJsObject(jsObject[key], service);
                    break;
                case XmlElementNames_1.XmlElementNames.Exceptions:
                    this.exceptions.LoadFromXmlJsObject(jsObject[key], service);
                    break;
                default:
                    break;
            }
        }
    }
    /**
     * @internal Writes the elements to XML writer.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        if (!ExtensionMethods_1.StringHelper.IsNullOrEmpty(this.Id)) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.RuleId, this.Id);
        }
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.DisplayName, this.DisplayName);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.Priority, this.Priority);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.IsEnabled, this.IsEnabled);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.IsInError, this.IsInError);
        this.Conditions.WriteToXml(writer, XmlElementNames_1.XmlElementNames.Conditions);
        this.Exceptions.WriteToXml(writer, XmlElementNames_1.XmlElementNames.Exceptions);
        this.Actions.WriteToXml(writer, XmlElementNames_1.XmlElementNames.Actions);
    }
}
exports.Rule = Rule;
//# sourceMappingURL=Rule.js.map