"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RuleActions = void 0;
const ExtensionMethods_1 = require("../ExtensionMethods");
const EmailAddress_1 = require("./EmailAddress");
const EmailAddressCollection_1 = require("./EmailAddressCollection");
const EwsUtilities_1 = require("../Core/EwsUtilities");
const FolderId_1 = require("./FolderId");
const Importance_1 = require("../Enumerations/Importance");
const ItemId_1 = require("./ItemId");
const MobilePhone_1 = require("../Misc/MobilePhone");
const StringList_1 = require("./StringList");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const XmlNamespace_1 = require("../Enumerations/XmlNamespace");
const ComplexProperty_1 = require("./ComplexProperty");
/**
 * Represents the set of actions available for a rule.
 *
 * @sealed
 */
class RuleActions extends ComplexProperty_1.ComplexProperty {
    /**
     * @internal Initializes a new instance of the **RuleActions** class.
     */
    constructor() {
        super();
        /**
         * The AssignCategories action.
         */
        this.assignCategories = null;
        /**
         * The CopyToFolder action.
         */
        this.copyToFolder = null;
        /**
         * The Delete action.
         */
        this.delete = false;
        /**
         * The ForwardAsAttachmentToRecipients action.
         */
        this.forwardAsAttachmentToRecipients = null;
        /**
         * The ForwardToRecipients action.
         */
        this.forwardToRecipients = null;
        /**
         * The MarkImportance action.
         */
        this.markImportance = null; //Nullable
        /**
         * The MarkAsRead action.
         */
        this.markAsRead = false;
        /**
         * The MoveToFolder action.
         */
        this.moveToFolder = null;
        /**
         * The PermanentDelete action.
         */
        this.permanentDelete = false;
        /**
         * The RedirectToRecipients action.
         */
        this.redirectToRecipients = null;
        /**
         * The SendSMSAlertToRecipients action.
         */
        this.sendSMSAlertToRecipients = null;
        /**
         * The ServerReplyWithMessage action.
         */
        this.serverReplyWithMessage = null;
        /**
         * The StopProcessingRules action.
         */
        this.stopProcessingRules = false;
        this.assignCategories = new StringList_1.StringList();
        this.forwardAsAttachmentToRecipients = new EmailAddressCollection_1.EmailAddressCollection(XmlElementNames_1.XmlElementNames.Address);
        this.forwardToRecipients = new EmailAddressCollection_1.EmailAddressCollection(XmlElementNames_1.XmlElementNames.Address);
        this.redirectToRecipients = new EmailAddressCollection_1.EmailAddressCollection(XmlElementNames_1.XmlElementNames.Address);
        this.sendSMSAlertToRecipients = [];
    }
    /**
     * Gets the categories that should be stamped on incoming messages.
     * To disable stamping incoming messages with categories, set AssignCategories to null.
     */
    get AssignCategories() {
        return this.assignCategories;
    }
    /**
     * Gets or sets the Id of the folder incoming messages should be copied to.
     * To disable copying incoming messages to a folder, set CopyToFolder to null.
     */
    get CopyToFolder() {
        return this.copyToFolder;
    }
    set CopyToFolder(value) {
        this.SetFieldValue({ getValue: () => this.copyToFolder, setValue: (updateValue) => { this.copyToFolder = updateValue; } }, value);
    }
    /**
     * Gets or sets a value indicating whether incoming messages should be automatically moved to the Deleted Items folder.
     */
    get Delete() {
        return this.delete;
    }
    set Delete(value) {
        this.SetFieldValue({ getValue: () => this.delete, setValue: (updateValue) => { this.delete = updateValue; } }, value);
    }
    /**
     * Gets the e-mail addresses to which incoming messages should be forwarded as attachments.
     * To disable forwarding incoming messages as attachments, empty the ForwardAsAttachmentToRecipients list.
     */
    get ForwardAsAttachmentToRecipients() {
        return this.forwardAsAttachmentToRecipients;
    }
    /**
     * Gets the e-mail addresses to which incoming messages should be forwarded.
     * To disable forwarding incoming messages, empty the ForwardToRecipients list.
     */
    get ForwardToRecipients() {
        return this.forwardToRecipients;
    }
    /**
     * @Nullable Gets or sets the importance that should be stamped on incoming messages.
     * To disable the stamping of incoming messages with an importance, set MarkImportance to null.
     */
    get MarkImportance() {
        return this.markImportance;
    }
    set MarkImportance(value) {
        this.SetFieldValue({ getValue: () => this.markImportance, setValue: (updateValue) => { this.markImportance = updateValue; } }, value);
    }
    /**
     * Gets or sets a value indicating whether incoming messages should be marked as read.
     */
    get MarkAsRead() {
        return this.markAsRead;
    }
    set MarkAsRead(value) {
        this.SetFieldValue({ getValue: () => this.markAsRead, setValue: (updateValue) => { this.markAsRead = updateValue; } }, value);
    }
    /**
     * Gets or sets the Id of the folder to which incoming messages should be moved.
     * To disable the moving of incoming messages to a folder, set CopyToFolder to null.
     */
    get MoveToFolder() {
        return this.moveToFolder;
    }
    set MoveToFolder(value) {
        this.SetFieldValue({ getValue: () => this.moveToFolder, setValue: (updateValue) => { this.moveToFolder = updateValue; } }, value);
    }
    /**
     * Gets or sets a value indicating whether incoming messages should be permanently deleted. When a message is permanently deleted, it is never saved into the recipient's mailbox. To delete a message after it has been saved into the recipient's mailbox, use the Delete action.
     */
    get PermanentDelete() {
        return this.permanentDelete;
    }
    set PermanentDelete(value) {
        this.SetFieldValue({ getValue: () => this.permanentDelete, setValue: (updateValue) => { this.permanentDelete = updateValue; } }, value);
    }
    /**
     * Gets the e-mail addresses to which incoming messages should be redirecteded.
     * To disable redirection of incoming messages, empty the RedirectToRecipients list. Unlike forwarded mail, redirected mail maintains the original sender and recipients.
     */
    get RedirectToRecipients() {
        return this.redirectToRecipients;
    }
    /**
     * Gets the phone numbers to which an SMS alert should be sent.
     * To disable sending SMS alerts for incoming messages, empty the SendSMSAlertToRecipients list.
     */
    get SendSMSAlertToRecipients() {
        return this.sendSMSAlertToRecipients;
    }
    /**
     * Gets or sets the Id of the template message that should be sent as a reply to incoming messages.
     * To disable automatic replies, set ServerReplyWithMessage to null.
     */
    get ServerReplyWithMessage() {
        return this.serverReplyWithMessage;
    }
    set ServerReplyWithMessage(value) {
        this.SetFieldValue({ getValue: () => this.serverReplyWithMessage, setValue: (updateValue) => { this.serverReplyWithMessage = updateValue; } }, value);
    }
    /**
     * Gets or sets a value indicating whether subsequent rules should be evaluated.
     */
    get StopProcessingRules() {
        return this.stopProcessingRules;
    }
    set StopProcessingRules(value) {
        this.SetFieldValue({ getValue: () => this.stopProcessingRules, setValue: (updateValue) => { this.stopProcessingRules = updateValue; } }, value);
    }
    /**
     * Convert the SMS recipient list from EmailAddressCollection type to MobilePhone collection type.
     *
     * @param   {EmailAddressCollection}   emailCollection   Recipient list in EmailAddressCollection type.
     * @return  {MobilePhone[]}		A MobilePhone collection object containing all SMS recipient in MobilePhone type.
     */
    static ConvertSMSRecipientsFromEmailAddressCollectionToMobilePhoneCollection(emailCollection) {
        let mobilePhoneCollection = [];
        for (let emailAddress of emailCollection.Items) {
            mobilePhoneCollection.push(new MobilePhone_1.MobilePhone(emailAddress.Name, emailAddress.Address));
        }
        return mobilePhoneCollection;
    }
    /**
     * Convert the SMS recipient list from MobilePhone collection type to EmailAddressCollection type.
     *
     * @param   {MobilePhone[]}   recipientCollection   Recipient list in a MobilePhone collection type.
     * @return  {EmailAddressCollection}		An EmailAddressCollection object containing recipients with "MOBILE" address type.
     */
    static ConvertSMSRecipientsFromMobilePhoneCollectionToEmailAddressCollection(recipientCollection) {
        let emailCollection = new EmailAddressCollection_1.EmailAddressCollection(XmlElementNames_1.XmlElementNames.Address);
        for (let recipient of recipientCollection) {
            let emailAddress = new EmailAddress_1.EmailAddress(recipient.Name, recipient.PhoneNumber, RuleActions.MobileType);
            emailCollection.Add(emailAddress);
        }
        return emailCollection;
    }
    /**
     * @internal Validates this instance.
     */
    InternalValidate() {
        super.InternalValidate();
        EwsUtilities_1.EwsUtilities.ValidateParam(this.forwardAsAttachmentToRecipients, "ForwardAsAttachmentToRecipients");
        EwsUtilities_1.EwsUtilities.ValidateParam(this.forwardToRecipients, "ForwardToRecipients");
        EwsUtilities_1.EwsUtilities.ValidateParam(this.redirectToRecipients, "RedirectToRecipients");
        for (let sendSMSAlertToRecipient of this.sendSMSAlertToRecipients) {
            EwsUtilities_1.EwsUtilities.ValidateParam(sendSMSAlertToRecipient, "SendSMSAlertToRecipient");
        }
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}                 jsObject                Json Object converted from XML.
     * @param   {ExchangeService}     service                 The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        for (let key in jsObject) {
            switch (key) {
                case XmlElementNames_1.XmlElementNames.AssignCategories:
                    this.assignCategories.LoadFromXmlJsObject(jsObject[key], service);
                    break;
                case XmlElementNames_1.XmlElementNames.CopyToFolder:
                    this.copyToFolder = new FolderId_1.FolderId();
                    this.copyToFolder.LoadFromXmlJsObject(jsObject[key][XmlElementNames_1.XmlElementNames.FolderId], service);
                    break;
                case XmlElementNames_1.XmlElementNames.Delete:
                    this.delete = ExtensionMethods_1.Convert.toBool(jsObject[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.ForwardAsAttachmentToRecipients:
                    this.forwardAsAttachmentToRecipients.LoadFromXmlJsObject(jsObject[key][XmlElementNames_1.XmlElementNames.Address], service);
                    break;
                case XmlElementNames_1.XmlElementNames.ForwardToRecipients:
                    this.forwardToRecipients.LoadFromXmlJsObject(jsObject[key][XmlElementNames_1.XmlElementNames.Address], service);
                    break;
                case XmlElementNames_1.XmlElementNames.MarkImportance:
                    this.markImportance = Importance_1.Importance[jsObject[key]];
                    break;
                case XmlElementNames_1.XmlElementNames.MarkAsRead:
                    this.markAsRead = ExtensionMethods_1.Convert.toBool(jsObject[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.MoveToFolder:
                    this.moveToFolder = new FolderId_1.FolderId();
                    this.moveToFolder.LoadFromXmlJsObject(jsObject[key][XmlElementNames_1.XmlElementNames.FolderId], service);
                    break;
                case XmlElementNames_1.XmlElementNames.PermanentDelete:
                    this.permanentDelete = ExtensionMethods_1.Convert.toBool(jsObject[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.RedirectToRecipients:
                    this.redirectToRecipients.LoadFromXmlJsObject(jsObject[key][XmlElementNames_1.XmlElementNames.Address], service);
                    break;
                case XmlElementNames_1.XmlElementNames.SendSMSAlertToRecipients:
                    let smsRecipientCollection = new EmailAddressCollection_1.EmailAddressCollection(XmlElementNames_1.XmlElementNames.Address);
                    smsRecipientCollection.LoadFromXmlJsObject(jsObject[key][XmlElementNames_1.XmlElementNames.Address], service);
                    this.sendSMSAlertToRecipients = RuleActions.ConvertSMSRecipientsFromEmailAddressCollectionToMobilePhoneCollection(smsRecipientCollection);
                    break;
                case XmlElementNames_1.XmlElementNames.ServerReplyWithMessage:
                    this.serverReplyWithMessage = new ItemId_1.ItemId();
                    this.serverReplyWithMessage.LoadFromXmlJsObject(jsObject[key][XmlElementNames_1.XmlElementNames.ItemId], service);
                    break;
                case XmlElementNames_1.XmlElementNames.StopProcessingRules:
                    this.stopProcessingRules = ExtensionMethods_1.Convert.toBool(jsObject[key]);
                    break;
                default:
                    break;
            }
        }
    }
    /**
     * @internal Writes the elements to XML writer.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        if (this.AssignCategories.Count > 0) {
            this.AssignCategories.WriteToXml(writer, XmlElementNames_1.XmlElementNames.AssignCategories);
        }
        if (this.CopyToFolder != null) {
            writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.CopyToFolder);
            this.CopyToFolder.WriteToXml(writer);
            writer.WriteEndElement();
        }
        if (this.Delete != false) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.Delete, this.Delete);
        }
        if (this.ForwardAsAttachmentToRecipients.Count > 0) {
            this.ForwardAsAttachmentToRecipients.WriteToXml(writer, XmlElementNames_1.XmlElementNames.ForwardAsAttachmentToRecipients);
        }
        if (this.ForwardToRecipients.Count > 0) {
            this.ForwardToRecipients.WriteToXml(writer, XmlElementNames_1.XmlElementNames.ForwardToRecipients);
        }
        if ((0, ExtensionMethods_1.hasValue)(this.MarkImportance)) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.MarkImportance, Importance_1.Importance[this.MarkImportance]);
        }
        if (this.MarkAsRead != false) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.MarkAsRead, this.MarkAsRead);
        }
        if (this.MoveToFolder != null) {
            writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.MoveToFolder);
            this.MoveToFolder.WriteToXml(writer);
            writer.WriteEndElement();
        }
        if (this.PermanentDelete != false) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.PermanentDelete, this.PermanentDelete);
        }
        if (this.RedirectToRecipients.Count > 0) {
            this.RedirectToRecipients.WriteToXml(writer, XmlElementNames_1.XmlElementNames.RedirectToRecipients);
        }
        if (this.SendSMSAlertToRecipients.length > 0) {
            let emailCollection = RuleActions.ConvertSMSRecipientsFromMobilePhoneCollectionToEmailAddressCollection(this.SendSMSAlertToRecipients);
            emailCollection.WriteToXml(writer, XmlElementNames_1.XmlElementNames.SendSMSAlertToRecipients);
        }
        if (this.ServerReplyWithMessage != null) {
            this.ServerReplyWithMessage.WriteToXml(writer, XmlElementNames_1.XmlElementNames.ServerReplyWithMessage);
        }
        if (this.StopProcessingRules != false) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.StopProcessingRules, this.StopProcessingRules);
        }
    }
}
exports.RuleActions = RuleActions;
/**
 * SMS recipient address type.
 */
RuleActions.MobileType = "MOBILE";
//# sourceMappingURL=RuleActions.js.map