"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RuleOperationError = void 0;
const ArgumentException_1 = require("../Exceptions/ArgumentException");
const ExtensionMethods_1 = require("../ExtensionMethods");
const RuleErrorCollection_1 = require("./RuleErrorCollection");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const ComplexProperty_1 = require("./ComplexProperty");
/**
 * Represents an error that occurred while processing a rule operation.
 *
 * @sealed
 */
class RuleOperationError extends ComplexProperty_1.ComplexProperty {
    /**
     * @internal Initializes a new instance of the **RuleOperationError** class.
     */
    constructor() {
        super();
        /**
         * Index of the operation mapping to the error.
         */
        this.operationIndex = 0;
        /**
         * RuleOperation object mapping to the error.
         */
        this.operation = null;
        /**
         * RuleError Collection.
         */
        this.ruleErrors = null;
    }
    /**
     * Gets the operation that resulted in an error.
     */
    get Operation() {
        return this.operation;
    }
    /**
     * Gets the number of rule errors in the list.
     */
    get Count() {
        return this.ruleErrors.Count;
    }
    /**
     * Gets the rule error at the specified index.
     *
     * @param   {number}   index   The index of the rule error to get.
     * @return  {RuleError}	The rule error at the specified index.
     */
    _getItem(index) {
        if (index < 0 || index >= this.ruleErrors.Count) {
            throw new ArgumentException_1.ArgumentOutOfRangeException("Index");
        }
        return this.ruleErrors._getItem(index);
    }
    /**
     *  Returns an enumerator that iterates through the collection. this case this.rules
     */
    GetEnumerator() {
        return this.ruleErrors.GetEnumerator();
    }
    /**
     * @internal Set operation property by the index of a given opeation enumerator.
     *
     * @param   {RuleOperation[]}   operations   Operation enumerator.
     */
    SetOperationByIndex(operations) {
        this.operation = operations[this.operationIndex];
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}                 jsObject                Json Object converted from XML.
     * @param   {ExchangeService}     service                 The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        for (let key in jsObject) {
            switch (key) {
                case XmlElementNames_1.XmlElementNames.OperationIndex:
                    this.operationIndex = ExtensionMethods_1.Convert.toNumber(jsObject[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.ValidationErrors:
                    this.ruleErrors = new RuleErrorCollection_1.RuleErrorCollection();
                    this.ruleErrors.CreateFromXmlJsObjectCollection(jsObject[key], service);
                    break;
                default:
                    break;
            }
        }
    }
}
exports.RuleOperationError = RuleOperationError;
//# sourceMappingURL=RuleOperationError.js.map