"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RulePredicateSizeRange = void 0;
const ExtensionMethods_1 = require("../ExtensionMethods");
const ServiceValidationException_1 = require("../Exceptions/ServiceValidationException");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const XmlNamespace_1 = require("../Enumerations/XmlNamespace");
const ComplexProperty_1 = require("./ComplexProperty");
/**
 * Represents the minimum and maximum size of a message.
 *
 * @sealed
 */
class RulePredicateSizeRange extends ComplexProperty_1.ComplexProperty {
    /**
     * @internal Initializes a new instance of the **RulePredicateSizeRange** class.
     */
    constructor() {
        super();
        /**
         * Minimum Size.
         */
        this.minimumSize = null; //Nullable
        /**
         * Mamixmum Size.
         */
        this.maximumSize = null; //Nullable
    }
    /**
     * @Nullable Gets or sets the minimum size, in kilobytes.
     * If MinimumSize is set to null, no minimum size applies.
     */
    get MinimumSize() {
        return this.minimumSize;
    }
    set MinimumSize(value) {
        this.SetFieldValue({ getValue: () => this.minimumSize, setValue: (updateValue) => { this.minimumSize = updateValue; } }, value);
    }
    /**
     * @Nullable Gets or sets the maximum size, in kilobytes.
     * If MaximumSize is set to null, no maximum size applies.
     */
    get MaximumSize() {
        return this.maximumSize;
    }
    set MaximumSize(value) {
        this.SetFieldValue({ getValue: () => this.maximumSize, setValue: (updateValue) => { this.maximumSize = updateValue; } }, value);
    }
    /**
     * @internal Validates this instance.
     */
    InternalValidate() {
        super.InternalValidate();
        if (this.minimumSize &&
            this.maximumSize &&
            this.minimumSize > this.maximumSize) {
            throw new ServiceValidationException_1.ServiceValidationException("MinimumSize cannot be larger than MaximumSize.");
        }
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}                 jsObject                Json Object converted from XML.
     * @param   {ExchangeService}     service                 The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        for (let key in jsObject) {
            switch (key) {
                case XmlElementNames_1.XmlElementNames.MinimumSize:
                    this.minimumSize = ExtensionMethods_1.Convert.toNumber(jsObject[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.MaximumSize:
                    this.maximumSize = ExtensionMethods_1.Convert.toNumber(jsObject[key]);
                    break;
                default:
                    break;
            }
        }
    }
    /**
     * @internal Writes the elements to XML writer.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        if (this.MinimumSize) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.MinimumSize, this.MinimumSize);
        }
        if (this.MaximumSize) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.MaximumSize, this.MaximumSize);
        }
    }
}
exports.RulePredicateSizeRange = RulePredicateSizeRange;
//# sourceMappingURL=RulePredicateSizeRange.js.map