"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StringList = void 0;
const ArgumentException_1 = require("../Exceptions/ArgumentException");
const ExtensionMethods_1 = require("../ExtensionMethods");
const Strings_1 = require("../Strings");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const XmlNamespace_1 = require("../Enumerations/XmlNamespace");
const ComplexProperty_1 = require("./ComplexProperty");
/**
 * Represents a list of strings.
 *
 * @sealed
 */
class StringList extends ComplexProperty_1.ComplexProperty {
    constructor(stringOrItemXmlElementName = null) {
        super();
        this.items = [];
        this.itemXmlElementName = XmlElementNames_1.XmlElementNames.String;
        if (stringOrItemXmlElementName !== null) {
            if (typeof stringOrItemXmlElementName === 'string') {
                this.itemXmlElementName = stringOrItemXmlElementName;
            }
            else {
                this.AddRange(stringOrItemXmlElementName);
            }
        }
    }
    /**
     * Gets the number of strings in the list.
     */
    get Count() {
        return this.items.length;
    }
    /**
     * Gets or sets the string at the specified index.
     *
     * @param   {number}    index   The index of the string to get or set.
     * @return  {string}    The string at the specified index.
     */
    _getItem(index) {
        if (index < 0 || index >= this.Count) {
            throw new ArgumentException_1.ArgumentOutOfRangeException("index - " + Strings_1.Strings.IndexIsOutOfRange);
        }
        return this.items[index];
    }
    /**
     * Gets or sets the string at the specified index.
     *
     * @param   {number}    index   The index of the string to get or set.
     * @return  {string}    The string at the specified index.
     */
    _setItem(index, value) {
        if (index < 0 || index >= this.Count) {
            throw new ArgumentException_1.ArgumentOutOfRangeException("index - " + Strings_1.Strings.IndexIsOutOfRange);
        }
        if (this.items[index] !== value) {
            this.items[index] = value;
            this.Changed();
        }
    }
    /**
     * Adds a string to the list.
     *
     * @param   {string}   s   The string to add.
     */
    Add(s) {
        this.items.push(s);
        this.Changed();
    }
    /**
     * Adds multiple strings to the list.
     *
     * @param   {string[]}   strings   The strings to add.
     */
    AddRange(strings) {
        let changed = false;
        for (let s of strings) {
            if (!this.Contains(s)) {
                this.items.push(s);
                changed = true;
            }
        }
        if (changed) {
            this.Changed();
        }
    }
    /**
     * Clears the list.
     */
    Clear() {
        this.items.splice(0);
        this.Changed();
    }
    /**
     * Determines whether the list contains a specific string.
     *
     * @param   {string}    s   The string to check the presence of.
     * @return  {boolean}   True if s is present in the list, false otherwise.
     */
    Contains(s) { return this.items.indexOf(s) >= 0; }
    /**
     * @internal Loads from XMLJsObject collection to create a new collection item.
     *
     * @interface   IJsonCollectionDeserializer
     *
     * @param   {any}               jsObjectCollection   The json collection.
     * @param   {ExchangeService}   service          The service.
     */
    CreateFromXmlJsObjectCollection(jsObjectCollection, service) {
        let collection = jsObjectCollection[this.itemXmlElementName];
        if (!ExtensionMethods_1.ArrayHelper.isArray(collection)) {
            collection = [collection];
        }
        for (let item of collection) {
            this.Add(item);
        }
    }
    Equals(obj) { throw new Error("StringList.ts - Equals : Not implemented."); }
    /**
     *  Returns an enumerator that iterates through the collection. this case this.items
     */
    GetEnumerator() {
        return this.items;
    }
    GetHashCode() { throw new Error("StringList.ts - GetHashCode : Not implemented."); }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}				jsObject	Json Object converted from XML.
     * @param   {ExchangeService}	service	The service.
     */
    LoadFromXmlJsObject(jsObjectCollection, service) {
        this.CreateFromXmlJsObjectCollection(jsObjectCollection, service);
    }
    /**
     * Removes a string from the list.
     *
     * @param   {string}   s   The string to remove.
     * @return  {boolean}       True is s was removed, false otherwise.
     */
    Remove(s) {
        let result = ExtensionMethods_1.ArrayHelper.RemoveEntry(this.items, s);
        if (result) {
            this.Changed();
        }
        return result;
    }
    /**
     * Removes the string at the specified position from the list.
     *
     * @param   {number}   index   The index of the string to remove.
     */
    RemoveAt(index) {
        if (index < 0 || index >= this.Count) {
            throw new ArgumentException_1.ArgumentOutOfRangeException("index - " + Strings_1.Strings.IndexIsOutOfRange);
        }
        this.items.splice(index, 1);
        this.Changed();
    }
    /**
     * Generates a string representation of all the items in the list.
     *
     * @return  {string}      A comma-separated list of the strings present in the list.
     */
    ToString() {
        return this.items.join(",");
    }
    /**
     * @internal Loads from XMLJsObject collection to update collection Items.
     *
     * @interface   IJsonCollectionDeserializer
     *
     * @param   {any}               jsObjectCollection   The XMLJsObject collection.
     * @param   {ExchangeService}   service          The service.
     */
    UpdateFromXmlJsObjectCollection(jsObjectCollection, service) {
        throw new Error("StringList.ts - UpdateFromXmlJsObjectCollection : Not implemented.");
    }
    /**
     * @internal Writes the elements to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        for (let item of this.items) {
            writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, this.itemXmlElementName);
            writer.WriteValue(item, this.itemXmlElementName);
            writer.WriteEndElement();
        }
    }
}
exports.StringList = StringList;
//# sourceMappingURL=StringList.js.map