"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TimeChange = void 0;
const DateTime_1 = require("../DateTime");
const EwsUtilities_1 = require("../Core/EwsUtilities");
const ExtensionMethods_1 = require("../ExtensionMethods");
const Time_1 = require("../Misc/Time");
const TimeChangeRecurrence_1 = require("./TimeChangeRecurrence");
const XmlAttributeNames_1 = require("../Core/XmlAttributeNames");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const XmlNamespace_1 = require("../Enumerations/XmlNamespace");
const ComplexProperty_1 = require("./ComplexProperty");
/**
 * @internal Represents a change of time for a time zone.
 * @sealed
 */
class TimeChange extends ComplexProperty_1.ComplexProperty {
    constructor(offset = null, time = null) {
        super();
        this.timeZoneName = null;
        this.offset = null;
        this.time = null;
        this.absoluteDate = null;
        this.recurrence = null;
        if (offset) {
            this.offset = offset;
        }
        if (time) {
            this.time = time;
        }
    }
    /**
     * Gets or sets the name of the associated time zone.
     */
    get TimeZoneName() {
        return this.timeZoneName;
    }
    set TimeZoneName(value) {
        this.SetFieldValue({ getValue: () => this.timeZoneName, setValue: (fieldValue) => { this.timeZoneName = fieldValue; } }, value);
    }
    /**
     * Gets or sets the offset since the beginning of the year when the change occurs.
     */
    get Offset() {
        return this.offset;
    }
    set Offset(value) {
        this.SetFieldValue({ getValue: () => this.offset, setValue: (fieldValue) => { this.offset = fieldValue; } }, value);
    }
    /**
     * Gets or sets the time at which the change occurs.
     */
    get Time() {
        return this.time;
    }
    set Time(value) {
        this.SetFieldValue({ getValue: () => this.time, setValue: (fieldValue) => { this.time = fieldValue; } }, value);
    }
    /**
     * Gets or sets the absolute date at which the change occurs. AbsoluteDate and Recurrence are mutually exclusive; setting one resets the other.
     */
    get AbsoluteDate() {
        return this.absoluteDate;
    }
    set AbsoluteDate(value) {
        this.SetFieldValue({ getValue: () => this.absoluteDate, setValue: (fieldValue) => { this.absoluteDate = fieldValue; } }, value);
        if (this.absoluteDate != null) {
            this.recurrence = null;
        }
    }
    /**
     * Gets or sets the recurrence pattern defining when the change occurs. Recurrence and AbsoluteDate are mutually exclusive; setting one resets the other.
     */
    get Recurrence() {
        return this.recurrence;
    }
    set Recurrence(value) {
        this.SetFieldValue({ getValue: () => this.recurrence, setValue: (fieldValue) => { this.recurrence = fieldValue; } }, value);
        if (this.recurrence != null) {
            this.absoluteDate = null;
        }
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}                 jsObject                Json Object converted from XML.
     * @param   {ExchangeService}     service                 The service.
     */
    LoadFromXmlJsObject(jsonProperty, service) {
        for (let key in jsonProperty) {
            switch (key) {
                case XmlElementNames_1.XmlElementNames.Offset:
                    this.offset = EwsUtilities_1.EwsUtilities.XSDurationToTimeSpan(jsonProperty[key]);
                case XmlElementNames_1.XmlElementNames.RelativeYearlyRecurrence:
                    this.Recurrence = new TimeChangeRecurrence_1.TimeChangeRecurrence();
                    this.Recurrence.LoadFromXmlJsObject(jsonProperty[key], service);
                case XmlElementNames_1.XmlElementNames.AbsoluteDate:
                    let dateTime = DateTime_1.DateTime.Parse(jsonProperty[key]);
                    // TODO: BUG
                    this.absoluteDate = new DateTime_1.DateTime(dateTime.ToUniversalTime().TotalMilliSeconds, DateTime_1.DateTimeKind.Unspecified);
                case XmlElementNames_1.XmlElementNames.Time:
                    this.time = new Time_1.Time(DateTime_1.DateTime.Parse(jsonProperty[key]));
                default:
            }
        }
    }
    /**
     * @internal Writes the attributes to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteAttributesToXml(writer) {
        writer.WriteAttributeValue(XmlAttributeNames_1.XmlAttributeNames.TimeZoneName, this.TimeZoneName);
    }
    /**
     * @internal Writes elements to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        if ((0, ExtensionMethods_1.hasValue)(this.Offset)) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.Offset, EwsUtilities_1.EwsUtilities.TimeSpanToXSDuration(this.Offset));
        }
        if (this.Recurrence != null) {
            this.Recurrence.WriteToXml(writer, XmlElementNames_1.XmlElementNames.RelativeYearlyRecurrence);
        }
        if (this.AbsoluteDate) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.AbsoluteDate, EwsUtilities_1.EwsUtilities.DateTimeToXSDate(new DateTime_1.DateTime(this.AbsoluteDate.TotalMilliSeconds, DateTime_1.DateTimeKind.Unspecified)));
        }
        if (this.Time != null) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.Time, this.Time.ToXSTime());
        }
    }
}
exports.TimeChange = TimeChange;
//# sourceMappingURL=TimeChange.js.map