"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AbsoluteDayOfMonthTransition = void 0;
const ExtensionMethods_1 = require("../../ExtensionMethods");
const DateTime_1 = require("../../DateTime");
const EwsLogging_1 = require("../../Core/EwsLogging");
const TimeZoneInfo_1 = require("../../TimeZoneInfo");
const XmlElementNames_1 = require("../../Core/XmlElementNames");
const XmlNamespace_1 = require("../../Enumerations/XmlNamespace");
const AbsoluteMonthTransition_1 = require("./AbsoluteMonthTransition");
/**
 * @internal Represents a time zone period transition that occurs on a specific day of a specific month.
 */
class AbsoluteDayOfMonthTransition extends AbsoluteMonthTransition_1.AbsoluteMonthTransition {
    constructor(timeZoneDefinition, targetPeriod) {
        super(timeZoneDefinition, targetPeriod);
        this.dayOfMonth = null;
    }
    /**
     * @internal Gets the day of then month when this transition occurs.
     */
    get DayOfMonth() {
        return this.dayOfMonth;
    }
    /**
     * @internal Creates a time zone transition time.
     * @virtual
     *
     * @return  {TimeZoneInfo.TransitionTime}      A TimeZoneInfo.TransitionTime.
     */
    CreateTransitionTime() {
        return TimeZoneInfo_1.TimeZoneInfo.TransitionTime.CreateFixedDateRule(new DateTime_1.DateTime(this.TimeOffset.Milliseconds), this.Month, this.DayOfMonth);
    }
    /**
     * @internal Gets the name of the XML element.
     *
     * @return  {string}      XML element name.
     */
    GetXmlElementName() {
        return XmlElementNames_1.XmlElementNames.RecurringDateTransition;
    }
    /**
     * @internal Initializes this transition based on the specified transition time.
     * @virtual
     *
     * @param   {TimeZoneInfo.TransitionTime}   transitionTime   The transition time to initialize from.
     */
    InitializeFromTransitionTime(transitionTime) {
        super.InitializeFromTransitionTime(transitionTime);
        this.dayOfMonth = transitionTime.Day;
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}				jsObject	Json Object converted from XML.
     * @param   {ExchangeService}	service	The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        super.LoadFromXmlJsObject(jsObject, service);
        if (jsObject[XmlElementNames_1.XmlElementNames.Day]) {
            this.dayOfMonth = ExtensionMethods_1.Convert.toInt(jsObject[XmlElementNames_1.XmlElementNames.Day]);
            EwsLogging_1.EwsLogging.Assert(this.dayOfMonth > 0 && this.dayOfMonth <= 31, "AbsoluteDayOfMonthTransition.TryReadElementFromXml", "dayOfMonth is not in the valid 1 - 31 range.");
        }
    }
    /**
     * @internal Writes elements to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        super.WriteElementsToXml(writer);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.Day, this.dayOfMonth);
    }
}
exports.AbsoluteDayOfMonthTransition = AbsoluteDayOfMonthTransition;
//# sourceMappingURL=AbsoluteDayOfMonthTransition.js.map