"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AbsoluteMonthTransition = void 0;
const ExtensionMethods_1 = require("../../ExtensionMethods");
const EwsLogging_1 = require("../../Core/EwsLogging");
const EwsUtilities_1 = require("../../Core/EwsUtilities");
const XmlElementNames_1 = require("../../Core/XmlElementNames");
const XmlNamespace_1 = require("../../Enumerations/XmlNamespace");
const TimeZoneTransition_1 = require("./TimeZoneTransition");
/**
 * @internal Represents the base class for all recurring time zone period transitions.
 */
class AbsoluteMonthTransition extends TimeZoneTransition_1.TimeZoneTransition {
    constructor(timeZoneDefinition, targetPeriod) {
        super(timeZoneDefinition, targetPeriod);
        this.timeOffset = null;
        this.month = null;
    }
    /**
     * @internal Gets the time offset from midnight when the transition occurs.
     */
    get TimeOffset() {
        return this.timeOffset;
    }
    /**
     * @internal Gets the month when the transition occurs.
     */
    get Month() {
        return this.month;
    }
    /**
     * @internal Initializes this transition based on the specified transition time.
     * @virtual
     *
     * @param   {TimeZoneInfo.TransitionTime}   transitionTime   The transition time to initialize from.
     */
    InitializeFromTransitionTime(transitionTime) {
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}				jsObject	Json Object converted from XML.
     * @param   {ExchangeService}	service	The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        super.LoadFromXmlJsObject(jsObject, service);
        if (jsObject[XmlElementNames_1.XmlElementNames.TimeOffset]) {
            this.timeOffset = EwsUtilities_1.EwsUtilities.XSDurationToTimeSpan(jsObject[XmlElementNames_1.XmlElementNames.TimeOffset]);
        }
        if (jsObject[XmlElementNames_1.XmlElementNames.Month]) {
            this.month = ExtensionMethods_1.Convert.toInt(jsObject[XmlElementNames_1.XmlElementNames.Month]);
            EwsLogging_1.EwsLogging.Assert(this.month > 0 && this.month <= 12, "AbsoluteMonthTransition.TryReadElementFromXml", "month is not in the valid 1 - 12 range.");
        }
    }
    /**
     * @internal Writes elements to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        super.WriteElementsToXml(writer);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.TimeOffset, EwsUtilities_1.EwsUtilities.TimeSpanToXSDuration(this.timeOffset));
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.Month, this.month);
    }
}
exports.AbsoluteMonthTransition = AbsoluteMonthTransition;
//# sourceMappingURL=AbsoluteMonthTransition.js.map