"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TimeZoneTransitionGroup = void 0;
const TimeSpan_1 = require("../../TimeSpan");
const TimeZoneInfo_1 = require("../../TimeZoneInfo");
const EwsServiceJsonReader_1 = require("../../Core/EwsServiceJsonReader");
const ServiceLocalException_1 = require("../../Exceptions/ServiceLocalException");
const ExtensionMethods_1 = require("../../ExtensionMethods");
const Strings_1 = require("../../Strings");
const TimeZonePeriod_1 = require("./TimeZonePeriod");
const TimeZoneTransition_1 = require("./TimeZoneTransition");
const XmlAttributeNames_1 = require("../../Core/XmlAttributeNames");
const XmlElementNames_1 = require("../../Core/XmlElementNames");
const ComplexProperty_1 = require("../ComplexProperty");
/**
 * @internal Represents a group of time zone period transitions.
 */
class TimeZoneTransitionGroup extends ComplexProperty_1.ComplexProperty {
    constructor(timeZoneDefinition, id) {
        super();
        //private id: string;
        this.transitions = [];
        this.timeZoneDefinition = timeZoneDefinition;
        this.Id = id;
    }
    /**
     * @internal Gets a value indicating whether this group contains a transition to the Daylight period.
     *
     * @value   *true* if this group contains a transition to daylight; otherwise, *false*.
     */
    get SupportsDaylight() {
        return this.transitions.length == 2;
    }
    /**
     * Gets the transition to the Daylight period.
     */
    get TransitionToDaylight() {
        this.InitializeTransitions();
        return this.transitionToDaylight;
    }
    /**
     * Gets the transition to the Standard period.
     */
    get TransitionToStandard() {
        this.InitializeTransitions();
        return this.transitionToStandard;
    }
    /**
     * @internal Gets the transitions in this group.
     */
    get Transitions() {
        return this.transitions;
    }
    /**
     * @internal Creates a time zone adjustment rule.
     *
     * @param   {DateTime}   startDate   The start date of the adjustment rule.
     * @param   {DateTime}   endDate     The end date of the adjustment rule.
     * @return  {TimeZoneInfo.AdjustmentRule}   An TimeZoneInfo.AdjustmentRule.
     */
    CreateAdjustmentRule(startDate, endDate) {
        // If there is only one transition, we can't create an adjustment rule. We have to assume
        // that the base offset to UTC is unchanged.
        if (this.transitions.length == 1) {
            return null;
        }
        return TimeZoneInfo_1.TimeZoneInfo.AdjustmentRule.CreateAdjustmentRule(startDate.Date, endDate.Date, this.GetDaylightDelta(), this.TransitionToDaylight.CreateTransitionTime(), this.TransitionToStandard.CreateTransitionTime());
    }
    /**
     * @internal Gets the offset to UTC based on this group's transitions.
     *
     * @returns {TimeZoneTransitionGroup.CustomTimeZoneCreateParams}
     */
    GetCustomTimeZoneCreationParams() {
        let result = new TimeZoneTransitionGroup.CustomTimeZoneCreateParams();
        if (this.TransitionToDaylight != null) {
            result.DaylightDisplayName = this.TransitionToDaylight.TargetPeriod.Name;
        }
        result.StandardDisplayName = this.TransitionToStandard.TargetPeriod.Name;
        // Assume that the standard period's offset is the base offset to UTC.
        // EWS returns a positive offset for time zones that are behind UTC, and
        // a negative one for time zones ahead of UTC. TimeZoneInfo does it the other
        // way around.
        result.BaseOffsetToUtc = new TimeSpan_1.TimeSpan(-this.TransitionToStandard.TargetPeriod.Bias);
        return result;
    }
    /**
     * @internal Gets the delta offset for the daylight.
     *
     * @returns {TimeSpan}
     */
    GetDaylightDelta() {
        if (this.SupportsDaylight) {
            // EWS returns a positive offset for time zones that are behind UTC, and
            // a negative one for time zones ahead of UTC. TimeZoneInfo does it the other
            // way around.
            return this.TransitionToStandard.TargetPeriod.Bias.Subtract(this.TransitionToDaylight.TargetPeriod.Bias);
        }
        else {
            return TimeSpan_1.TimeSpan.Zero;
        }
    }
    /**
     * Initializes this transition group based on the specified asjustment rule.
     *
     * @param   {TimeZoneInfo.AdjustmentRule}   adjustmentRule   The adjustment rule to initialize from.
     * @param   {TimeZonePeriod}                standardPeriod   A reference to the pre-created standard period.
     */
    InitializeFromAdjustmentRule(adjustmentRule, standardPeriod) {
        let daylightPeriod = new TimeZonePeriod_1.TimeZonePeriod();
        // Generate an Id of the form "Daylight/2008"
        daylightPeriod.Id = ExtensionMethods_1.StringHelper.Format("{0}/{1}", TimeZonePeriod_1.TimeZonePeriod.DaylightPeriodId, adjustmentRule.DateStart.Year);
        daylightPeriod.Name = TimeZonePeriod_1.TimeZonePeriod.DaylightPeriodName;
        daylightPeriod.Bias = new TimeSpan_1.TimeSpan(standardPeriod.Bias.TotalMilliseconds - adjustmentRule.DaylightDelta.TotalMilliseconds);
        this.timeZoneDefinition.Periods.Add(daylightPeriod.Id, daylightPeriod);
        this.transitionToDaylight = TimeZoneTransition_1.TimeZoneTransition.CreateTimeZoneTransition(this.timeZoneDefinition, daylightPeriod, adjustmentRule.DaylightTransitionStart);
        let standardPeriodToSet = new TimeZonePeriod_1.TimeZonePeriod();
        standardPeriodToSet.Id = ExtensionMethods_1.StringHelper.Format("{0}/{1}", standardPeriod.Id, adjustmentRule.DateStart.Year);
        standardPeriodToSet.Name = standardPeriod.Name;
        standardPeriodToSet.Bias = standardPeriod.Bias;
        this.timeZoneDefinition.Periods.Add(standardPeriodToSet.Id, standardPeriodToSet);
        this.transitionToStandard = TimeZoneTransition_1.TimeZoneTransition.CreateTimeZoneTransition(this.timeZoneDefinition, standardPeriodToSet, adjustmentRule.DaylightTransitionEnd);
        this.transitions.push(this.transitionToDaylight);
        this.transitions.push(this.transitionToStandard);
    }
    /**
     * Initializes the private members holding references to the transitions to the Daylight and Standard periods.
     */
    InitializeTransitions() {
        if (this.transitionToStandard == null) {
            for (let transition of this.transitions) {
                if (transition.TargetPeriod.IsStandardPeriod || (this.transitions.length == 1)) {
                    this.transitionToStandard = transition;
                }
                else {
                    this.transitionToDaylight = transition;
                }
            }
        }
        // If we didn't find a Standard period, this is an invalid time zone group.
        if (this.transitionToStandard == null) {
            throw new ServiceLocalException_1.ServiceLocalException(Strings_1.Strings.InvalidOrUnsupportedTimeZoneDefinition);
        }
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}				jsObject	Json Object converted from XML.
     * @param   {ExchangeService}	service	The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        if (jsObject[XmlElementNames_1.XmlElementNames.Id]) {
            this.Id = jsObject[XmlElementNames_1.XmlElementNames.Id];
        }
        for (let key in jsObject) {
            if (TimeZoneTransitionGroup.transitionTypes.indexOf(key) >= 0) {
                let transitions = EwsServiceJsonReader_1.EwsServiceJsonReader.ReadAsArray(jsObject, key);
                for (let item of transitions) {
                    let transition = TimeZoneTransition_1.TimeZoneTransition.Create(this.timeZoneDefinition, key);
                    transition.LoadFromXmlJsObject(item, service);
                    this.transitions.push(transition);
                }
            }
        }
    }
    /**
     * @internal Validates this transition group.
     */
    Validate() {
        // There must be exactly one or two transitions in the group.
        if (this.transitions.length < 1 || this.transitions.length > 2) {
            throw new ServiceLocalException_1.ServiceLocalException(Strings_1.Strings.InvalidOrUnsupportedTimeZoneDefinition);
        }
        // If there is only one transition, it must be of type TimeZoneTransition
        if (this.transitions.length == 1 && !(this.transitions[0] instanceof TimeZoneTransition_1.TimeZoneTransition)) {
            throw new ServiceLocalException_1.ServiceLocalException(Strings_1.Strings.InvalidOrUnsupportedTimeZoneDefinition);
        }
        // If there are two transitions, none of them should be of type TimeZoneTransition
        if (this.transitions.length == 2) {
            for (let transition of this.transitions) {
                try {
                    let type = transition.constructor.name;
                    //if (transition instanceof TimeZoneTransition) { // ref:  can not use due to prototype chain issue
                    if (type === "TimeZoneTransition") {
                        throw new ServiceLocalException_1.ServiceLocalException(Strings_1.Strings.InvalidOrUnsupportedTimeZoneDefinition);
                    }
                }
                catch (error) {
                }
            }
        }
        // All the transitions in the group must be to a period.
        for (let transition of this.transitions) {
            if (transition.TargetPeriod == null) {
                throw new ServiceLocalException_1.ServiceLocalException(Strings_1.Strings.InvalidOrUnsupportedTimeZoneDefinition);
            }
        }
    }
    /**
     * @internal Writes the attributes to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteAttributesToXml(writer) {
        writer.WriteAttributeValue(XmlAttributeNames_1.XmlAttributeNames.Id, this.Id);
    }
    /**
     * @internal Writes elements to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        for (let transition of this.transitions) {
            transition.WriteToXml(writer);
        }
    }
    /**
     * @internal Writes to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteToXml(writer) {
        super.WriteToXml(writer, XmlElementNames_1.XmlElementNames.TransitionsGroup);
    }
}
exports.TimeZoneTransitionGroup = TimeZoneTransitionGroup;
/** @internal */
TimeZoneTransitionGroup.transitionTypes = [XmlElementNames_1.XmlElementNames.AbsoluteDateTransition, XmlElementNames_1.XmlElementNames.RecurringDayTransition, XmlElementNames_1.XmlElementNames.RecurringDateTransition, XmlElementNames_1.XmlElementNames.Transition];
(function (TimeZoneTransitionGroup) {
    /**
     * @internal Represents custom time zone creation parameters.
     */
    class CustomTimeZoneCreateParams {
        /**
         * @internal Initializes a new instance of the **CustomTimeZoneCreateParams** class.
         */
        constructor() {
            this.baseOffsetToUtc = null;
            this.standardDisplayName = null;
            this.daylightDisplayName = null;
        }
        /**
         * @internal Gets or sets the base offset to UTC.
         */
        get BaseOffsetToUtc() {
            return this.baseOffsetToUtc;
        }
        set BaseOffsetToUtc(value) {
            this.baseOffsetToUtc = value;
        }
        /**
         * @internal Gets or sets the display name of the standard period.
         */
        get StandardDisplayName() {
            return this.standardDisplayName;
        }
        set StandardDisplayName(value) {
            this.standardDisplayName = value;
        }
        /**
         * @internal Gets or sets the display name of the daylight period.
         */
        get DaylightDisplayName() {
            return this.daylightDisplayName;
        }
        set DaylightDisplayName(value) {
            this.daylightDisplayName = value;
        }
        /**
         * @internal Gets a value indicating whether the custom time zone should have a daylight period.
         *
         * @value   *true* if the custom time zone should have a daylight period; otherwise, *false*.
         */
        get HasDaylightPeriod() {
            return !ExtensionMethods_1.StringHelper.IsNullOrEmpty(this.daylightDisplayName);
        }
    }
    TimeZoneTransitionGroup.CustomTimeZoneCreateParams = CustomTimeZoneCreateParams;
})(TimeZoneTransitionGroup = exports.TimeZoneTransitionGroup || (exports.TimeZoneTransitionGroup = {}));
//# sourceMappingURL=TimeZoneTransitionGroup.js.map