"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConvertIdRequest = void 0;
const EwsUtilities_1 = require("../EwsUtilities");
const ExchangeVersion_1 = require("../../Enumerations/ExchangeVersion");
const IdFormat_1 = require("../../Enumerations/IdFormat");
const XmlAttributeNames_1 = require("../XmlAttributeNames");
const XmlElementNames_1 = require("../XmlElementNames");
const XmlNamespace_1 = require("../../Enumerations/XmlNamespace");
const ConvertIdResponse_1 = require("../Responses/ConvertIdResponse");
const MultiResponseServiceRequest_1 = require("./MultiResponseServiceRequest");
/**
 * @internal Represents a ConvertId request.
 *
 * @sealed
 */
class ConvertIdRequest extends MultiResponseServiceRequest_1.MultiResponseServiceRequest {
    /**
     * @internal Initializes a new instance of the **ConvertIdRequest** class.
     *
     * @param   {ExchangeService}   service             The service.
     * @param   {ServiceErrorHandling}   errorHandlingMode   Indicates how errors should be handled.
     */
    constructor(service, errorHandlingMode) {
        super(service, errorHandlingMode);
        this.destinationFormat = IdFormat_1.IdFormat.EwsId;
        this.ids = [];
    }
    /**
     * Gets or sets the destination format.
     *
     * @value   The destination format.
     */
    get DestinationFormat() {
        return this.destinationFormat;
    }
    set DestinationFormat(value) {
        this.destinationFormat = value;
    }
    /**
     * Gets the ids.
     *
     * @value   The ids.
     */
    get Ids() {
        return this.ids;
    }
    /**
     * @internal Creates the service response.
     *
     * @param   {ExchangeService}   service         The service.
     * @param   {number}            responseIndex   Index of the response.
     * @return  {ConvertIdResponse}     Service response.
     */
    CreateServiceResponse(service, responseIndex) {
        return new ConvertIdResponse_1.ConvertIdResponse();
    }
    /**
     * @internal Gets the expected response message count.
     *
     * @return  {number}      Number of expected response messages.
     */
    GetExpectedResponseMessageCount() {
        return this.Ids.length;
    }
    /**
     * @internal Gets the request version.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this request is supported.
     */
    GetMinimumRequiredServerVersion() {
        return ExchangeVersion_1.ExchangeVersion.Exchange2007_SP1;
    }
    /**
     * @internal Gets the name of the response message XML element.
     *
     * @return  {string}      XML element name,
     */
    GetResponseMessageXmlElementName() {
        return XmlElementNames_1.XmlElementNames.ConvertIdResponseMessage;
    }
    /**
     * @internal Gets the name of the response XML element.
     *
     * @return  {string}      XML element name,
     */
    GetResponseXmlElementName() {
        return XmlElementNames_1.XmlElementNames.ConvertIdResponse;
    }
    /**
     * @internal Gets the name of the XML element.
     *
     * @return  {string}      XML element name,
     */
    GetXmlElementName() {
        return XmlElementNames_1.XmlElementNames.ConvertId;
    }
    /**
     * @internal Validate request.
     */
    Validate() {
        super.Validate();
        EwsUtilities_1.EwsUtilities.ValidateParamCollection(this.Ids, "Ids");
    }
    /**
     * @internal Writes the elements to XML writer.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        writer.WriteAttributeValue(XmlAttributeNames_1.XmlAttributeNames.DestinationFormat, IdFormat_1.IdFormat[this.DestinationFormat]);
        writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.SourceIds);
        for (let alternateId of this.Ids) {
            alternateId.WriteToXml(writer);
        }
        writer.WriteEndElement(); // SourceIds
    }
}
exports.ConvertIdRequest = ConvertIdRequest;
//# sourceMappingURL=ConvertIdRequest.js.map