"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GetAttachmentRequest = void 0;
const ArgumentException_1 = require("../../Exceptions/ArgumentException");
const BodyType_1 = require("../../Enumerations/BodyType");
const EwsUtilities_1 = require("../EwsUtilities");
const ExchangeVersion_1 = require("../../Enumerations/ExchangeVersion");
const PropertySet_1 = require("../PropertySet");
const Schemas_1 = require("../ServiceObjects/Schemas/Schemas");
const ExtensionMethods_1 = require("../../ExtensionMethods");
const Strings_1 = require("../../Strings");
const XmlAttributeNames_1 = require("../XmlAttributeNames");
const XmlElementNames_1 = require("../XmlElementNames");
const XmlNamespace_1 = require("../../Enumerations/XmlNamespace");
const GetAttachmentResponse_1 = require("../Responses/GetAttachmentResponse");
const MultiResponseServiceRequest_1 = require("./MultiResponseServiceRequest");
/**
 * @internal Represents a GetAttachment request.
 */
class GetAttachmentRequest extends MultiResponseServiceRequest_1.MultiResponseServiceRequest {
    /**
     * @internal Initializes a new instance of the **GetAttachmentRequest** class.
     *
     * @param   {ExchangeService}       service             The service.
     * @param   {ServiceErrorHandling}  errorHandlingMode   Indicates how errors should be handled.
     */
    constructor(service, errorHandlingMode) {
        super(service, errorHandlingMode);
        this.attachments = [];
        this.attachmentIds = [];
        this.additionalProperties = [];
        this.bodyType = null;
    }
    /**
     * Gets the attachments.
     *
     * @value The attachments.
     */
    get Attachments() {
        return this.attachments;
    }
    /**
     * Gets the attachment ids.
     *
     * @value The attachment ids.
     */
    get AttachmentIds() {
        return this.attachmentIds;
    }
    /**
     * Gets the additional properties.
     *
     * @value The additional properties.
     */
    get AdditionalProperties() {
        return this.additionalProperties;
    }
    /**
     * Gets or sets the type of the body.
     *
     * @value The type of the body.
     */
    get BodyType() {
        return this.bodyType;
    }
    set BodyType(value) {
        this.bodyType = value;
    }
    /**
     * @internal Gets a value indicating whether the TimeZoneContext SOAP header should be emitted.
     *
     * @value *true* if the time zone should be emitted; otherwise, *false*.
     */
    get EmitTimeZoneHeader() {
        return this.additionalProperties.indexOf(Schemas_1.Schemas.ItemSchema.MimeContent) >= 0;
    }
    /**
     * @internal Creates the service response.
     *
     * @param   {ExchangeService}   service         The service.
     * @param   {number}            responseIndex   Index of the response.
     * @return  {GetAttachmentResponse}             Service response.
     */
    CreateServiceResponse(service, responseIndex) {
        return new GetAttachmentResponse_1.GetAttachmentResponse(this.Attachments.length > 0 ? this.Attachments[responseIndex] : null);
    }
    /**
     * @internal Gets the expected response message count.
     *
     * @return  {number}      Number of expected response messages.
     */
    GetExpectedResponseMessageCount() { return this.Attachments.length + this.AttachmentIds.length; }
    /**
     * @internal Gets the request version.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this request is supported.
     */
    GetMinimumRequiredServerVersion() { return ExchangeVersion_1.ExchangeVersion.Exchange2007_SP1; }
    /**
     * @internal Gets the name of the response message XML element.
     *
     * @return  {string}      XML element name,
     */
    GetResponseMessageXmlElementName() { return XmlElementNames_1.XmlElementNames.GetAttachmentResponseMessage; }
    /**
     * @internal Gets the name of the response XML element.
     *
     * @return  {string}      XML element name,
     */
    GetResponseXmlElementName() { return XmlElementNames_1.XmlElementNames.GetAttachmentResponse; }
    /**
     * @internal Gets the name of the XML element.
     *
     * @return  {string}      XML element name,
     */
    GetXmlElementName() { return XmlElementNames_1.XmlElementNames.GetAttachment; }
    /**
     * @internal Validate request.
     */
    Validate() {
        super.Validate();
        if (this.Attachments.length > 0) {
            EwsUtilities_1.EwsUtilities.ValidateParamCollection(this.Attachments, "Attachments");
        }
        if (this.AttachmentIds.length > 0) {
            EwsUtilities_1.EwsUtilities.ValidateParamCollection(this.AttachmentIds, "AttachmentIds");
        }
        if (this.AttachmentIds.length == 0 && this.Attachments.length == 0) {
            throw new ArgumentException_1.ArgumentException(Strings_1.Strings.CollectionIsEmpty, "Attachments/AttachmentIds");
        }
        for (let i = 0; i < this.AdditionalProperties.length; i++) {
            EwsUtilities_1.EwsUtilities.ValidateParam(this.AdditionalProperties[i], ExtensionMethods_1.StringHelper.Format("AdditionalProperties[{0}]", i));
        }
    }
    /**
     * @internal Writes attachment id elements.
     *
     * @param   {EwsServiceXmlWriter}   writer         The writer.
     * @param   {string}                attachmentId   The attachment id.
     */
    WriteAttachmentIdXml(writer, attachmentId) {
        writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.AttachmentId);
        writer.WriteAttributeValue(XmlAttributeNames_1.XmlAttributeNames.Id, attachmentId);
        writer.WriteEndElement();
    }
    /**
     * @internal Writes XML elements.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        if (this.BodyType || this.AdditionalProperties.length > 0) {
            writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.AttachmentShape);
            if ((0, ExtensionMethods_1.hasValue)(this.BodyType)) {
                writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.BodyType, BodyType_1.BodyType[this.BodyType]);
            }
            if (this.AdditionalProperties.length > 0) {
                PropertySet_1.PropertySet.WriteAdditionalPropertiesToXml(writer, this.AdditionalProperties);
            }
            writer.WriteEndElement(); // AttachmentShape
        }
        writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.AttachmentIds);
        for (let attachment of this.Attachments) {
            this.WriteAttachmentIdXml(writer, attachment.Id);
        }
        for (let attachmentId of this.AttachmentIds) {
            this.WriteAttachmentIdXml(writer, attachmentId);
        }
        writer.WriteEndElement();
    }
}
exports.GetAttachmentRequest = GetAttachmentRequest;
//# sourceMappingURL=GetAttachmentRequest.js.map