"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GetUserConfigurationRequest = void 0;
const ExtensionMethods_1 = require("../../ExtensionMethods");
const EwsUtilities_1 = require("../EwsUtilities");
const ExchangeVersion_1 = require("../../Enumerations/ExchangeVersion");
const ServiceErrorHandling_1 = require("../../Enumerations/ServiceErrorHandling");
const UserConfiguration_1 = require("../../Misc/UserConfiguration");
const UserConfigurationProperties_1 = require("../../Enumerations/UserConfigurationProperties");
const XmlElementNames_1 = require("../XmlElementNames");
const XmlNamespace_1 = require("../../Enumerations/XmlNamespace");
const GetUserConfigurationResponse_1 = require("../Responses/GetUserConfigurationResponse");
const MultiResponseServiceRequest_1 = require("./MultiResponseServiceRequest");
/**
 * @internal Represents a GetUserConfiguration request.
 */
class GetUserConfigurationRequest extends MultiResponseServiceRequest_1.MultiResponseServiceRequest {
    /**
     * @internal Initializes a new instance of the **GetUserConfigurationRequest** class.
     *
     * @param   {ExchangeService}       service   The service.
     */
    constructor(service) {
        super(service, ServiceErrorHandling_1.ServiceErrorHandling.ThrowOnError);
        this.name = null;
        this.parentFolderId = null;
        this.properties = UserConfigurationProperties_1.UserConfigurationProperties.Id;
        this.userConfiguration = null;
    }
    /**
     * @internal Gets or sets the name.
     *
     * @value   The name.
     */
    get Name() {
        return this.name;
    }
    set Name(value) {
        this.name = value;
    }
    /**
     * @internal Gets or sets the parent folder Id.
     *
     * @value   The parent folder Id.
     */
    get ParentFolderId() {
        return this.parentFolderId;
    }
    set ParentFolderId(value) {
        this.parentFolderId = value;
    }
    /**
     * @internal Gets or sets the user configuration.
     *
     * @value   The userConfiguration.
     */
    get UserConfiguration() {
        return this.userConfiguration;
    }
    set UserConfiguration(value) {
        this.userConfiguration = value;
        this.name = this.userConfiguration.Name;
        this.parentFolderId = this.userConfiguration.ParentFolderId;
    }
    /**
     * @internal Gets or sets the properties.
     *
     * @value   The properties.
     */
    get Properties() {
        return this.properties;
    }
    set Properties(value) {
        this.properties = value;
    }
    /**
     * @internal Creates the service response.
     *
     * @param   {ExchangeService}   service         The service.
     * @param   {number}   			responseIndex   Index of the response.
     * @return  {GetUserConfigurationResponse}	    Service response.
     */
    CreateServiceResponse(service, responseIndex) {
        // In the case of UserConfiguration.Load(), this.userConfiguration is set.
        if (this.userConfiguration == null) {
            this.userConfiguration = new UserConfiguration_1.UserConfiguration(service, this.properties);
            this.userConfiguration.Name = this.name;
            this.userConfiguration.ParentFolderId = this.parentFolderId;
        }
        return new GetUserConfigurationResponse_1.GetUserConfigurationResponse(this.userConfiguration);
    }
    /**
     * @internal Gets the expected response message count.
     *
     * @return  {number}      Number of expected response messages.
     */
    GetExpectedResponseMessageCount() {
        return 1;
    }
    /**
     * @internal Gets the request version.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this request is supported.
     */
    GetMinimumRequiredServerVersion() {
        return ExchangeVersion_1.ExchangeVersion.Exchange2010;
    }
    /**
     * @internal Gets the name of the response message XML element.
     *
     * @return  {string}      Xml element name.
     */
    GetResponseMessageXmlElementName() {
        return XmlElementNames_1.XmlElementNames.GetUserConfigurationResponseMessage;
    }
    /**
     * @internal Gets the name of the response XML element.
     *
     * @return  {string}      Xml element name.
     */
    GetResponseXmlElementName() {
        return XmlElementNames_1.XmlElementNames.GetUserConfigurationResponse;
    }
    /**
     * @internal Gets the name of the XML element.
     *
     * @return  {string}      Xml element name.
     */
    GetXmlElementName() {
        return XmlElementNames_1.XmlElementNames.GetUserConfiguration;
    }
    /**
     * @internal Validate the request.
     */
    Validate() {
        super.Validate();
        EwsUtilities_1.EwsUtilities.ValidateParam(this.name, "name");
        EwsUtilities_1.EwsUtilities.ValidateParam(this.parentFolderId, "parentFolderId");
        this.ParentFolderId.Validate(this.Service.RequestedServerVersion);
    }
    /**
     * @internal Writes XML elements.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        // Write UserConfiguationName element
        UserConfiguration_1.UserConfiguration.WriteUserConfigurationNameToXml(writer, XmlNamespace_1.XmlNamespace.Messages, this.name, this.parentFolderId);
        // Write UserConfigurationProperties element
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.UserConfigurationProperties, this.properties === UserConfigurationProperties_1.UserConfigurationProperties.All ? UserConfigurationProperties_1.UserConfigurationProperties[this.properties] : ExtensionMethods_1.EnumHelper.ToString(UserConfigurationProperties_1.UserConfigurationProperties, this.properties).split(GetUserConfigurationRequest.EnumDelimiter).join(""));
        //.replace(GetUserConfigurationRequest.EnumDelimiter, " ")); //info: replace only replaces first occurance
    }
}
exports.GetUserConfigurationRequest = GetUserConfigurationRequest;
GetUserConfigurationRequest.EnumDelimiter = ",";
//# sourceMappingURL=GetUserConfigurationRequest.js.map