"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SearchMailboxesRequest = void 0;
const Versioning_1 = require("../../MailboxSearch/Versioning");
const EwsServiceJsonReader_1 = require("../EwsServiceJsonReader");
const ExchangeVersion_1 = require("../../Enumerations/ExchangeVersion");
const MailboxSearchLocation_1 = require("../../Enumerations/MailboxSearchLocation");
const MailboxSearchScopeType_1 = require("../../Enumerations/MailboxSearchScopeType");
const PreviewItemBaseShape_1 = require("../../Enumerations/PreviewItemBaseShape");
const SearchPageDirection_1 = require("../../Enumerations/SearchPageDirection");
const SearchResultType_1 = require("../../Enumerations/SearchResultType");
const ServiceObjectSchema_1 = require("../ServiceObjects/Schemas/ServiceObjectSchema");
const ServiceResponseCollection_1 = require("../Responses/ServiceResponseCollection");
const ServiceValidationException_1 = require("../../Exceptions/ServiceValidationException");
const ServiceVersionException_1 = require("../../Exceptions/ServiceVersionException");
const SortDirection_1 = require("../../Enumerations/SortDirection");
const ExtensionMethods_1 = require("../../ExtensionMethods");
const Strings_1 = require("../../Strings");
const XmlElementNames_1 = require("../XmlElementNames");
const XmlNamespace_1 = require("../../Enumerations/XmlNamespace");
const SearchMailboxesResponse_1 = require("../Responses/SearchMailboxesResponse");
const MultiResponseServiceRequest_1 = require("./MultiResponseServiceRequest");
/**
 * @internal Represents a SearchMailboxesRequest request.
 *
 * @sealed
 */
class SearchMailboxesRequest extends MultiResponseServiceRequest_1.MultiResponseServiceRequest {
    /**
     * @internal Initializes a new instance of the **SearchMailboxesRequest** class.
     *
     * @param   {ExchangeService}       service   The service.
     * @param   {ServiceErrorHandling}  errorHandlingMode   Indicates how errors should be handled.
     */
    constructor(service, errorHandlingMode) {
        super(service, errorHandlingMode);
        this.searchQueries = [];
        this.searchResultType = SearchResultType_1.SearchResultType.PreviewOnly;
        this.sortOrder = SortDirection_1.SortDirection.Ascending;
        this.sortByProperty = null;
        this.performDeduplication = false;
        this.pageSize = 0;
        this.pageItemReference = null;
        this.pageDirection = SearchPageDirection_1.SearchPageDirection.Next;
        this.previewItemResponseShape = null;
        /**
         * Query language
         */
        this.Language = null;
        /**
         * Gets or sets the server version.
         * @interface IDiscoveryVersionable
         */
        this.ServerVersion = 0;
    }
    /**
     * Collection of query + mailboxes
     */
    get SearchQueries() {
        return this.searchQueries;
    }
    set SearchQueries(value) {
        this.searchQueries = value;
    }
    /**
     * Search result type
     */
    get ResultType() {
        return this.searchResultType;
    }
    set ResultType(value) {
        this.searchResultType = value;
    }
    /**
     * Preview item response shape
     */
    get PreviewItemResponseShape() {
        return this.previewItemResponseShape;
    }
    set PreviewItemResponseShape(value) {
        this.previewItemResponseShape = value;
    }
    /**
     * Sort order
     */
    get SortOrder() {
        return this.sortOrder;
    }
    set SortOrder(value) {
        this.sortOrder = value;
    }
    /**
     * Sort by property name
     */
    get SortByProperty() {
        return this.sortByProperty;
    }
    set SortByProperty(value) {
        this.sortByProperty = value;
    }
    /**
     * Perform deduplication or not
     */
    get PerformDeduplication() {
        return this.performDeduplication;
    }
    set PerformDeduplication(value) {
        this.performDeduplication = value;
    }
    /**
     * Page size
     */
    get PageSize() {
        return this.pageSize;
    }
    set PageSize(value) {
        this.pageSize = value;
    }
    /**
     * Page item reference
     */
    get PageItemReference() {
        return this.pageItemReference;
    }
    set PageItemReference(value) {
        this.pageItemReference = value;
    }
    /**
     * Page direction
     */
    get PageDirection() {
        return this.pageDirection;
    }
    set PageDirection(value) {
        this.pageDirection = value;
    }
    /**
     * @internal Creates the service response.
     *
     * @param   {ExchangeService}   service         The service.
     * @param   {number}   			responseIndex   Index of the response.
     * @return  {SearchMailboxesResponse}	Service response.
     */
    CreateServiceResponse(service, responseIndex) {
        return new SearchMailboxesResponse_1.SearchMailboxesResponse();
    }
    /**
     * @internal Gets the expected response message count.
     *
     * @return  {number}      Number of expected response messages.
     */
    GetExpectedResponseMessageCount() {
        return 1;
    }
    /**
     * @internal Gets the request version.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this request is supported.
     */
    GetMinimumRequiredServerVersion() {
        return ExchangeVersion_1.ExchangeVersion.Exchange2013;
    }
    /**
     * @internal Gets the name of the response message XML element.
     *
     * @return  {string}      Xml element name.
     */
    GetResponseMessageXmlElementName() {
        return XmlElementNames_1.XmlElementNames.SearchMailboxesResponseMessage;
    }
    /**
     * @internal Gets the name of the response XML element.
     *
     * @return  {string}      Xml element name.
     */
    GetResponseXmlElementName() {
        return XmlElementNames_1.XmlElementNames.SearchMailboxesResponse;
    }
    /**
     * @internal Gets the name of the XML element.
     *
     * @return  {string}      Xml element name.
     */
    GetXmlElementName() {
        return XmlElementNames_1.XmlElementNames.SearchMailboxes;
    }
    /**
     * @internal Parses the response.
     * See O15:324151 (OfficeDev bug/issue not visible to external world) on why we need to override ParseResponse here instead of calling the one in MultiResponseServiceRequest.cs
     *
     * @param   {any}   jsonBody   The js object response body.
     * @return  {any}              Response object.
     */
    ParseResponse(jsonBody) {
        let serviceResponses = new ServiceResponseCollection_1.ServiceResponseCollection();
        let jsResponseMessages = EwsServiceJsonReader_1.EwsServiceJsonReader.ReadAsArray(jsonBody[XmlElementNames_1.XmlElementNames.ResponseMessages], this.GetResponseMessageXmlElementName());
        for (let jsResponseObject of jsResponseMessages) {
            let response = new SearchMailboxesResponse_1.SearchMailboxesResponse();
            response.LoadFromXmlJsObject(jsResponseObject, this.Service);
            serviceResponses.Add(response);
        }
        return serviceResponses;
    }
    /**
     * @internal Validate the request.
     */
    Validate() {
        super.Validate();
        if (this.SearchQueries == null || this.SearchQueries.length == 0) {
            throw new ServiceValidationException_1.ServiceValidationException(Strings_1.Strings.MailboxQueriesParameterIsNotSpecified);
        }
        for (let searchQuery of this.SearchQueries) {
            if (searchQuery.MailboxSearchScopes == null || searchQuery.MailboxSearchScopes.length == 0) {
                throw new ServiceValidationException_1.ServiceValidationException(Strings_1.Strings.MailboxQueriesParameterIsNotSpecified);
            }
            for (let searchScope of searchQuery.MailboxSearchScopes) {
                if (searchScope.ExtendedAttributes != null && searchScope.ExtendedAttributes.length > 0 && !Versioning_1.DiscoverySchemaChanges.SearchMailboxesExtendedData.IsCompatible(this)) {
                    throw new ServiceVersionException_1.ServiceVersionException(ExtensionMethods_1.StringHelper.Format(Strings_1.Strings.ClassIncompatibleWithRequestVersion, "ExtendedAttribute", //typeof (ExtendedAttribute).Name,
                    Versioning_1.DiscoverySchemaChanges.SearchMailboxesExtendedData.MinimumServerVersion));
                }
                if (searchScope.SearchScopeType != MailboxSearchScopeType_1.MailboxSearchScopeType.LegacyExchangeDN && (!Versioning_1.DiscoverySchemaChanges.SearchMailboxesExtendedData.IsCompatible(this) || !Versioning_1.DiscoverySchemaChanges.SearchMailboxesAdditionalSearchScopes.IsCompatible(this))) {
                    throw new ServiceVersionException_1.ServiceVersionException(ExtensionMethods_1.StringHelper.Format(Strings_1.Strings.EnumValueIncompatibleWithRequestVersion, MailboxSearchScopeType_1.MailboxSearchScopeType[searchScope.SearchScopeType], "MailboxSearchScopeType", //typeof (MailboxSearchScopeType).Name,
                    Versioning_1.DiscoverySchemaChanges.SearchMailboxesAdditionalSearchScopes.MinimumServerVersion));
                }
            }
        }
        if (!ExtensionMethods_1.StringHelper.IsNullOrEmpty(this.SortByProperty)) {
            let prop = null;
            try {
                prop = ServiceObjectSchema_1.ServiceObjectSchema.FindPropertyDefinition(this.SortByProperty);
            }
            catch (ex) { //KeyNotFoundException
            }
            if (prop == null) {
                throw new ServiceValidationException_1.ServiceValidationException(ExtensionMethods_1.StringHelper.Format(Strings_1.Strings.InvalidSortByPropertyForMailboxSearch, this.SortByProperty));
            }
        }
    }
    /**
     * @internal Writes XML elements.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.SearchQueries);
        for (let mailboxQuery of this.SearchQueries) {
            writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.MailboxQuery);
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.Query, mailboxQuery.Query);
            writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.MailboxSearchScopes);
            for (let mailboxSearchScope of mailboxQuery.MailboxSearchScopes) {
                // The checks here silently downgrade the schema based on compatability checks, to recieve errors use the validate method
                if (mailboxSearchScope.SearchScopeType == MailboxSearchScopeType_1.MailboxSearchScopeType.LegacyExchangeDN || Versioning_1.DiscoverySchemaChanges.SearchMailboxesAdditionalSearchScopes.IsCompatible(this)) {
                    writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.MailboxSearchScope);
                    writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.Mailbox, mailboxSearchScope.Mailbox);
                    writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.SearchScope, MailboxSearchLocation_1.MailboxSearchLocation[mailboxSearchScope.SearchScope]);
                    if (Versioning_1.DiscoverySchemaChanges.SearchMailboxesExtendedData.IsCompatible(this)) {
                        writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.ExtendedAttributes);
                        if (mailboxSearchScope.SearchScopeType != MailboxSearchScopeType_1.MailboxSearchScopeType.LegacyExchangeDN) {
                            writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.ExtendedAttribute);
                            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.ExtendedAttributeName, XmlElementNames_1.XmlElementNames.SearchScopeType);
                            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.ExtendedAttributeValue, MailboxSearchScopeType_1.MailboxSearchScopeType[mailboxSearchScope.SearchScopeType]);
                            writer.WriteEndElement();
                        }
                        if (mailboxSearchScope.ExtendedAttributes != null && mailboxSearchScope.ExtendedAttributes.length > 0) {
                            for (let attribute of mailboxSearchScope.ExtendedAttributes) {
                                writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.ExtendedAttribute);
                                writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.ExtendedAttributeName, attribute.Name);
                                writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.ExtendedAttributeValue, attribute.Value);
                                writer.WriteEndElement();
                            }
                        }
                        writer.WriteEndElement(); // ExtendedData
                    }
                    writer.WriteEndElement(); // MailboxSearchScope
                }
            }
            writer.WriteEndElement(); // MailboxSearchScopes
            writer.WriteEndElement(); // MailboxQuery
        }
        writer.WriteEndElement(); // SearchQueries
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.ResultType, SearchResultType_1.SearchResultType[this.ResultType]);
        if (this.PreviewItemResponseShape != null) {
            writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.PreviewItemResponseShape);
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.BaseShape, PreviewItemBaseShape_1.PreviewItemBaseShape[this.PreviewItemResponseShape.BaseShape]);
            if (this.PreviewItemResponseShape.AdditionalProperties != null && this.PreviewItemResponseShape.AdditionalProperties.length > 0) {
                writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.AdditionalProperties);
                for (let additionalProperty of this.PreviewItemResponseShape.AdditionalProperties) {
                    additionalProperty.WriteToXml(writer);
                }
                writer.WriteEndElement(); // AdditionalProperties
            }
            writer.WriteEndElement(); // PreviewItemResponseShape
        }
        if (!ExtensionMethods_1.StringHelper.IsNullOrEmpty(this.SortByProperty)) {
            writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.SortBy);
            writer.WriteAttributeValue(XmlElementNames_1.XmlElementNames.Order, SortDirection_1.SortDirection[this.SortOrder]);
            writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.FieldURI);
            writer.WriteAttributeValue(XmlElementNames_1.XmlElementNames.FieldURI, this.sortByProperty);
            writer.WriteEndElement(); // FieldURI
            writer.WriteEndElement(); // SortBy
        }
        // Language
        if (!ExtensionMethods_1.StringHelper.IsNullOrEmpty(this.Language)) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.Language, this.Language);
        }
        // Dedupe
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.Deduplication, this.performDeduplication);
        if (this.PageSize > 0) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.PageSize, this.PageSize);
        }
        if (!ExtensionMethods_1.StringHelper.IsNullOrEmpty(this.PageItemReference)) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.PageItemReference, this.PageItemReference);
        }
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.PageDirection, SearchPageDirection_1.SearchPageDirection[this.PageDirection]);
    }
}
exports.SearchMailboxesRequest = SearchMailboxesRequest;
//# sourceMappingURL=SearchMailboxesRequest.js.map