"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SubscribeRequest = void 0;
const ExtensionMethods_1 = require("../../ExtensionMethods");
const EventType_1 = require("../../Enumerations/EventType");
const EwsUtilities_1 = require("../EwsUtilities");
const FolderIdWrapperList_1 = require("../../Misc/FolderIdWrapperList");
const ServiceErrorHandling_1 = require("../../Enumerations/ServiceErrorHandling");
const ServiceValidationException_1 = require("../../Exceptions/ServiceValidationException");
const Strings_1 = require("../../Strings");
const XmlAttributeNames_1 = require("../XmlAttributeNames");
const XmlElementNames_1 = require("../XmlElementNames");
const XmlNamespace_1 = require("../../Enumerations/XmlNamespace");
const MultiResponseServiceRequest_1 = require("./MultiResponseServiceRequest");
/**
 * @internal Represents an abstract Subscribe request.
 *
 * @typeparam	{TSubscription}		The type of the subscription.
 */
class SubscribeRequest extends MultiResponseServiceRequest_1.MultiResponseServiceRequest {
    /**
     * @internal Initializes a new instance of the **SubscribeRequest<TSubscription>** class.
     *
     * @param   {ExchangeService}   service   The service.
     */
    constructor(service) {
        super(service, ServiceErrorHandling_1.ServiceErrorHandling.ThrowOnError);
        this.FolderIds = new FolderIdWrapperList_1.FolderIdWrapperList();
        this.EventTypes = [];
    }
    /**
     * @internal Gets the expected response message count.
     *
     * @return  {number}      Number of expected response messages.
     */
    GetExpectedResponseMessageCount() { return 1; }
    /**
     * @internal Gets the name of the response message XML element.
     *
     * @return  {string}      XML element name,
     */
    GetResponseMessageXmlElementName() { return XmlElementNames_1.XmlElementNames.SubscribeResponseMessage; }
    /**
     * @internal Gets the name of the response XML element.
     *
     * @return  {string}      XML element name,
     */
    GetResponseXmlElementName() { return XmlElementNames_1.XmlElementNames.SubscribeResponse; }
    /**
     * @internal Gets the name of the XML element.
     *
     * @return  {string}      XML element name,
     */
    GetXmlElementName() { return XmlElementNames_1.XmlElementNames.Subscribe; }
    /**
     * @internal Validate request.
     */
    Validate() {
        super.Validate();
        EwsUtilities_1.EwsUtilities.ValidateParam(this.FolderIds, "FolderIds");
        EwsUtilities_1.EwsUtilities.ValidateParamCollection(this.EventTypes, "EventTypes");
        this.FolderIds.Validate(this.Service.RequestedServerVersion);
        // Check that caller isn't trying to subscribe to Status events.
        if (ExtensionMethods_1.ArrayHelper.OfType(this.EventTypes, (eventType) => eventType === EventType_1.EventType.Status).length > 0) { //  this.EventTypes.Count<EventType>(eventType => (eventType == EventType.Status)) > 0)
            throw new ServiceValidationException_1.ServiceValidationException(Strings_1.Strings.CannotSubscribeToStatusEvents);
        }
        // If Watermark was specified, make sure it's not a blank string.
        if (!ExtensionMethods_1.StringHelper.IsNullOrEmpty(this.Watermark)) {
            EwsUtilities_1.EwsUtilities.ValidateNonBlankStringParam(this.Watermark, "Watermark");
        }
        this.EventTypes.forEach((eventType) => {
            EwsUtilities_1.EwsUtilities.ValidateEnumVersionValue(EventType_1.EventType, eventType, this.Service.RequestedServerVersion, "EventType");
        });
    }
    /**
     * @internal Writes XML elements.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Messages, this.GetSubscriptionXmlElementName());
        if (this.FolderIds.Count == 0) {
            writer.WriteAttributeValue(XmlAttributeNames_1.XmlAttributeNames.SubscribeToAllFolders, true);
        }
        this.FolderIds.WriteToXml(writer, XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.FolderIds);
        writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.EventTypes);
        for (let eventType of this.EventTypes) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.EventType, EventType_1.EventType[eventType] + "Event");
        }
        writer.WriteEndElement();
        if (!ExtensionMethods_1.StringHelper.IsNullOrEmpty(this.Watermark)) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.Watermark, this.Watermark);
        }
        this.InternalWriteElementsToXml(writer);
        writer.WriteEndElement();
    }
}
exports.SubscribeRequest = SubscribeRequest;
//# sourceMappingURL=SubscribeRequest.js.map