"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UpdateFolderRequest = void 0;
const EwsUtilities_1 = require("../EwsUtilities");
const ExchangeVersion_1 = require("../../Enumerations/ExchangeVersion");
const ExtensionMethods_1 = require("../../ExtensionMethods");
const Strings_1 = require("../../Strings");
const UpdateFolderResponse_1 = require("../Responses/UpdateFolderResponse");
const XmlElementNames_1 = require("../XmlElementNames");
const XmlNamespace_1 = require("../../Enumerations/XmlNamespace");
const MultiResponseServiceRequest_1 = require("./MultiResponseServiceRequest");
/**
 * @internal Represents an UpdateFolder request.
 *
 * @sealed
*/
class UpdateFolderRequest extends MultiResponseServiceRequest_1.MultiResponseServiceRequest {
    /**
     * @internal Initializes a new instance of the **UpdateFolderRequest** class.
     *
     * @param   {ExchangeService}       service             The service.
     * @param   {ServiceErrorHandling}  errorHandlingMode   Indicates how errors should be handled.
     */
    constructor(service, errorHandlingMode) {
        super(service, errorHandlingMode);
        this.folders = [];
    }
    /**
     * Gets the list of folders.
     *
     * @value   The folders.
     */
    get Folders() {
        return this.folders;
    }
    /**
     * @internal Creates the service response.
     *
     * @param   {ExchangeService}   session         The session.
     * @param   {number}            responseIndex   Index of the response.
     * @return  {ServiceResponse}   Service response.
     */
    CreateServiceResponse(session, responseIndex) {
        return new UpdateFolderResponse_1.UpdateFolderResponse(this.Folders[responseIndex]);
    }
    /**
     * @internal Gets the expected response message count.
     *
     * @return  {number}      Number of expected response messages.
     */
    GetExpectedResponseMessageCount() {
        return this.folders.length;
    }
    /**
     * @internal Gets the request version.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this request is supported.
     */
    GetMinimumRequiredServerVersion() {
        return ExchangeVersion_1.ExchangeVersion.Exchange2007_SP1;
    }
    /**
     * @internal Gets the name of the response message XML element.
     *
     * @return  {string}      XML element name,
     */
    GetResponseMessageXmlElementName() {
        return XmlElementNames_1.XmlElementNames.UpdateFolderResponseMessage;
    }
    /**
     * @internal Gets the name of the response XML element.
     *
     * @return  {string}      XML element name,
     */
    GetResponseXmlElementName() {
        return XmlElementNames_1.XmlElementNames.UpdateFolderResponse;
    }
    /**
     * @internal Gets the name of the XML element.
     *
     * @return  {string}      XML element name,
     */
    GetXmlElementName() {
        return XmlElementNames_1.XmlElementNames.UpdateFolder;
    }
    /**
     * @internal Validates the request.
     */
    Validate() {
        super.Validate();
        EwsUtilities_1.EwsUtilities.ValidateParamCollection(this.Folders, "Folders");
        for (var folder of this.folders) {
            if ((folder == null) || folder.IsNew) {
                throw new Error(ExtensionMethods_1.StringHelper.Format(Strings_1.Strings.FolderToUpdateCannotBeNullOrNew, this.folders.indexOf(folder)));
            }
            folder.Validate();
        }
    }
    /**
     * @internal Writes XML elements.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.FolderChanges);
        for (var folder of this.folders) {
            folder.WriteToXmlForUpdate(writer);
        }
        writer.WriteEndElement();
    }
}
exports.UpdateFolderRequest = UpdateFolderRequest;
//# sourceMappingURL=UpdateFolderRequest.js.map