"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UpdateInboxRulesRequest = void 0;
const ArgumentException_1 = require("../../Exceptions/ArgumentException");
const EwsUtilities_1 = require("../EwsUtilities");
const ExchangeVersion_1 = require("../../Enumerations/ExchangeVersion");
const ServiceResult_1 = require("../../Enumerations/ServiceResult");
const ExtensionMethods_1 = require("../../ExtensionMethods");
const UpdateInboxRulesException_1 = require("../../Exceptions/UpdateInboxRulesException");
const UpdateInboxRulesResponse_1 = require("../Responses/UpdateInboxRulesResponse");
const XmlElementNames_1 = require("../XmlElementNames");
const XmlNamespace_1 = require("../../Enumerations/XmlNamespace");
const SimpleServiceRequestBase_1 = require("./SimpleServiceRequestBase");
/**
 * @internal Represents a UpdateInboxRulesRequest request.
 *
 * @sealed
 */
class UpdateInboxRulesRequest extends SimpleServiceRequestBase_1.SimpleServiceRequestBase {
    /**
     * @internal Initializes a new instance of the **UpdateInboxRulesRequest** class.
     *
     * @param   {ExchangeService}   service   The service.
     */
    constructor(service) {
        super(service);
        /**
         * The smtp address of the mailbox from which to get the inbox rules.
         */
        this.mailboxSmtpAddress = null;
        /**
         * Remove OutlookRuleBlob or not.
         */
        this.removeOutlookRuleBlob = false;
        /**
         * InboxRule operation collection.
         */
        this.inboxRuleOperations = null;
    }
    get MailboxSmtpAddress() {
        return this.mailboxSmtpAddress;
    }
    set MailboxSmtpAddress(value) {
        this.mailboxSmtpAddress = value;
    }
    get RemoveOutlookRuleBlob() {
        return this.removeOutlookRuleBlob;
    }
    set RemoveOutlookRuleBlob(value) {
        this.removeOutlookRuleBlob = value;
    }
    get InboxRuleOperations() {
        return this.inboxRuleOperations;
    }
    set InboxRuleOperations(value) {
        this.inboxRuleOperations = value;
    }
    /**
     * @internal Executes this request.
     *
     * @return  {Promise<UpdateInboxRulesResponse>}      Service response  :Promise.
     */
    Execute() {
        return this.InternalExecute().then((serviceResponse) => {
            if (serviceResponse.Result == ServiceResult_1.ServiceResult.Error) {
                throw new UpdateInboxRulesException_1.UpdateInboxRulesException(serviceResponse, this.inboxRuleOperations);
            }
            return serviceResponse;
        });
    }
    /**
     * @internal Gets the request version.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this request is supported.
     */
    GetMinimumRequiredServerVersion() {
        return ExchangeVersion_1.ExchangeVersion.Exchange2010_SP1;
    }
    /**
     * @internal Gets the name of the response XML element.
     *
     * @return  {string}      XML element name,
     */
    GetResponseXmlElementName() {
        return XmlElementNames_1.XmlElementNames.UpdateInboxRulesResponse;
    }
    /**
     * @internal Gets the name of the XML element.
     *
     * @return  {string}      XML element name,
     */
    GetXmlElementName() {
        return XmlElementNames_1.XmlElementNames.UpdateInboxRules;
    }
    /**
     * @internal Parses the response.
     *
     * @param   {any}   jsonBody   The js object response body.
     * @return  {any}              Response object.
     */
    ParseResponse(jsonBody) {
        let response = new UpdateInboxRulesResponse_1.UpdateInboxRulesResponse();
        response.LoadFromXmlJsObject(jsonBody, this.Service);
        return response;
    }
    /**
     * @internal Validate request.
     */
    Validate() {
        if (this.inboxRuleOperations == null) {
            throw new ArgumentException_1.ArgumentException("RuleOperations cannot be null.", "Operations");
        }
        let operationCount = 0;
        for (let operation of this.inboxRuleOperations) {
            EwsUtilities_1.EwsUtilities.ValidateParam(operation, "RuleOperation");
            operationCount++;
        }
        if (operationCount == 0) {
            throw new ArgumentException_1.ArgumentException("RuleOperations cannot be empty.", "Operations");
        }
        this.Service.Validate();
    }
    /**
     * @internal Writes the elements to XML writer.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        if (!ExtensionMethods_1.StringHelper.IsNullOrEmpty(this.mailboxSmtpAddress)) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.MailboxSmtpAddress, this.mailboxSmtpAddress);
        }
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.RemoveOutlookRuleBlob, this.RemoveOutlookRuleBlob);
        writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.Operations);
        for (let operation of this.inboxRuleOperations) {
            operation.WriteToXml(writer, operation.XmlElementName);
        }
        writer.WriteEndElement();
    }
}
exports.UpdateInboxRulesRequest = UpdateInboxRulesRequest;
//# sourceMappingURL=UpdateInboxRulesRequest.js.map