"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SyncResponse = void 0;
const ChangeCollection_1 = require("../../Sync/ChangeCollection");
const ChangeType_1 = require("../../Enumerations/ChangeType");
const ExtensionMethods_1 = require("../../ExtensionMethods");
const EwsLogging_1 = require("../EwsLogging");
const EwsServiceJsonReader_1 = require("../EwsServiceJsonReader");
const FolderInfo_1 = require("../ServiceObjects/Folders/FolderInfo");
const ItemChange_1 = require("../../Sync/ItemChange");
const ItemInfo_1 = require("../ServiceObjects/Items/ItemInfo");
const ServiceObjectInfo_1 = require("../ServiceObjects/ServiceObjectInfo");
const XmlElementNames_1 = require("../XmlElementNames");
const ServiceResponse_1 = require("./ServiceResponse");
/**
 * Represents the base response class for synchronuization operations.
 *
 * @sealed
 * @typeparam	{TServiceObject}	ServiceObject type.
 * @typeparam	{TChange}	Change type.
 */
class SyncResponse extends ServiceResponse_1.ServiceResponse {
    /**
     * @internal Initializes a new instance of the **SyncResponse<TServiceObject, TChange>** class.
     *
     * @param   {PropertySet}   propertySet   Property set.
     */
    constructor(propertySet) {
        super();
        this.changes = new ChangeCollection_1.ChangeCollection();
        this.propertySet = null;
        this.propertySet = propertySet;
        EwsLogging_1.EwsLogging.Assert(this.propertySet != null, "SyncResponse.ctor", "PropertySet should not be null");
    }
    /**
     * Gets a list of changes that occurred on the synchronized folder.
     */
    get Changes() {
        return this.changes;
    }
    /**
     * @internal Gets a value indicating whether this request returns full or summary properties.
     */
    get SummaryPropertiesOnly() { return false; /* abstract */ }
    /**
     * @internal Reads response elements from Xml JsObject.
     *
     * @param   {any}               jsObject   The response object.
     * @param   {ExchangeService}   service    The service.
     */
    ReadElementsFromXmlJsObject(responseObject, service) {
        this.Changes.SyncState = responseObject[XmlElementNames_1.XmlElementNames.SyncState];
        this.Changes.MoreChangesAvailable = !ExtensionMethods_1.Convert.toBool(responseObject[this.GetIncludesLastInRangeXmlElementName()]);
        let changesElement = responseObject[XmlElementNames_1.XmlElementNames.Changes];
        let folderInfo = new FolderInfo_1.FolderInfo();
        let itemInfo = new ItemInfo_1.ItemInfo();
        for (let changeElementKey in changesElement) {
            if (changeElementKey.indexOf("__") === 0)
                continue;
            let changeObjs = EwsServiceJsonReader_1.EwsServiceJsonReader.ReadAsArray(changesElement, changeElementKey);
            for (let changeObj of changeObjs) {
                let change = this.CreateChangeInstance();
                //let changeType:string = jsChange.ReadAsString(XmlElementNames.ChangeType);
                switch (changeElementKey) {
                    case XmlElementNames_1.XmlElementNames.Create:
                        change.ChangeType = ChangeType_1.ChangeType.Create;
                        break;
                    case XmlElementNames_1.XmlElementNames.Update:
                        change.ChangeType = ChangeType_1.ChangeType.Update;
                        break;
                    case XmlElementNames_1.XmlElementNames.Delete:
                        change.ChangeType = ChangeType_1.ChangeType.Delete;
                        break;
                    case XmlElementNames_1.XmlElementNames.ReadFlagChange:
                        change.ChangeType = ChangeType_1.ChangeType.ReadFlagChange;
                        break;
                    default:
                        break;
                }
                let changeObjectTypeName = ExtensionMethods_1.TypeSystem.GetJsObjectOnlyChildName(changeObj);
                let serviceObjectInfo = ServiceObjectInfo_1.ServiceObjectInfo.IsFolderType(changeObjectTypeName) ? folderInfo : itemInfo;
                if (change != null) {
                    let jsServiceObject = changeObj[changeObjectTypeName];
                    switch (change.ChangeType) {
                        case ChangeType_1.ChangeType.Delete:
                        case ChangeType_1.ChangeType.ReadFlagChange:
                            change.Id = change.CreateId();
                            let jsChangeId = changeObj[this.GetChangeIdElementName()];
                            change.Id.LoadFromXmlJsObject(jsChangeId, service);
                            if (change.ChangeType == ChangeType_1.ChangeType.ReadFlagChange) {
                                let itemChange = change;
                                EwsLogging_1.EwsLogging.Assert(change instanceof ItemChange_1.ItemChange, "SyncResponse.ReadElementsFromXmlJsObject", "ReadFlagChange is only valid on ItemChange");
                                itemChange.IsRead = ExtensionMethods_1.Convert.toBool(changeObj[XmlElementNames_1.XmlElementNames.IsRead]);
                            }
                            break;
                        default:
                            change.ServiceObject = serviceObjectInfo.CreateEwsObjectFromXmlElementName(service, changeObjectTypeName);
                            change.ServiceObject.LoadFromXmlJsObject(jsServiceObject, service, true, /* clearPropertyBag */ this.propertySet, this.SummaryPropertiesOnly);
                            break;
                    }
                    this.changes.Add(change);
                }
            }
        }
    }
}
exports.SyncResponse = SyncResponse;
//# sourceMappingURL=SyncResponse.js.map