"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Folder = void 0;
const EwsLogging_1 = require("../../EwsLogging");
const ExchangeVersion_1 = require("../../../Enumerations/ExchangeVersion");
const FolderId_1 = require("../../../ComplexProperties/FolderId");
const FolderView_1 = require("../../../Search/FolderView");
const Grouping_1 = require("../../../Search/Grouping");
const ItemView_1 = require("../../../Search/ItemView");
const PropertySet_1 = require("../../PropertySet");
const Schemas_1 = require("../Schemas/Schemas");
const SearchFilter_1 = require("../../../Search/Filters/SearchFilter");
const ServiceErrorHandling_1 = require("../../../Enumerations/ServiceErrorHandling");
const ServiceObjectSchema_1 = require("../Schemas/ServiceObjectSchema");
const ViewBase_1 = require("../../../Search/ViewBase");
const WellKnownFolderName_1 = require("../../../Enumerations/WellKnownFolderName");
const XmlElementNames_1 = require("../../XmlElementNames");
const ServiceObject_1 = require("../ServiceObject");
class Folder extends ServiceObject_1.ServiceObject {
    /**
     * Gets the Id of the folder.
     *
     */
    get Id() { return this.PropertyBag._getItem(this.GetIdPropertyDefinition()); }
    /**
     * Gets the Id of this folder's parent folder.
     *
     */
    get ParentFolderId() { return this.PropertyBag._getItem(Schemas_1.Schemas.FolderSchema.ParentFolderId); }
    /**
     * Gets the number of child folders this folder has.
     *
     */
    get ChildFolderCount() { return this.PropertyBag._getItem(Schemas_1.Schemas.FolderSchema.ChildFolderCount); }
    /**
     * Gets or sets the display name of the folder.
     *
     */
    get DisplayName() { return this.PropertyBag._getItem(Schemas_1.Schemas.FolderSchema.DisplayName); }
    set DisplayName(value) { this.PropertyBag._setItem(Schemas_1.Schemas.FolderSchema.DisplayName, value); }
    /**
     * Gets or sets the custom class name of this folder.
     *
     */
    get FolderClass() { return this.PropertyBag._getItem(Schemas_1.Schemas.FolderSchema.FolderClass); }
    set FolderClass(value) { this.PropertyBag._setItem(Schemas_1.Schemas.FolderSchema.FolderClass, value); }
    /**
     * Gets the total number of items contained in the folder.
     *
     */
    get TotalCount() { return this.PropertyBag._getItem(Schemas_1.Schemas.FolderSchema.TotalCount); }
    /**
     * Gets a list of extended properties associated with the folder. **Unstable Need testing**
     *
     */
    get ExtendedProperties() { return this.PropertyBag._getItem(ServiceObjectSchema_1.ServiceObjectSchema.ExtendedProperties); }
    /**
     * Gets the Email Lifecycle Management (ELC) information associated with the folder.
     *
     */
    get ManagedFolderInformation() { return this.PropertyBag._getItem(Schemas_1.Schemas.FolderSchema.ManagedFolderInformation); }
    /**
     * Gets a value indicating the effective rights the current authenticated user has on the folder.
     *
     */
    get EffectiveRights() { return this.PropertyBag._getItem(Schemas_1.Schemas.FolderSchema.EffectiveRights); }
    /**
     * Gets a list of permissions for the folder.
     *
     */
    get Permissions() { return this.PropertyBag._getItem(Schemas_1.Schemas.FolderSchema.Permissions); }
    /**
     * Gets the number of unread items in the folder.
     *
     */
    get UnreadCount() { return this.PropertyBag._getItem(Schemas_1.Schemas.FolderSchema.UnreadCount); }
    /**
     * Gets or sets the policy tag.
     *
     */
    get PolicyTag() { return this.PropertyBag._getItem(Schemas_1.Schemas.FolderSchema.PolicyTag); }
    set PolicyTag(value) { this.PropertyBag._setItem(Schemas_1.Schemas.FolderSchema.PolicyTag, value); }
    /**
     * Gets or sets the archive tag.
     *
     */
    get ArchiveTag() { return this.PropertyBag._getItem(Schemas_1.Schemas.FolderSchema.ArchiveTag); }
    set ArchiveTag(value) { this.PropertyBag._setItem(Schemas_1.Schemas.FolderSchema.ArchiveTag, value); }
    /**
     * Gets the well known name of this folder, if any, as a string.
     * **value** - The well known name of this folder as a string, or null if this folder isn't a well known folder.
     *
     */
    get WellKnownFolderNameAsString() { return WellKnownFolderName_1.WellKnownFolderName[this.PropertyBag._getItem(Schemas_1.Schemas.FolderSchema.WellKnownFolderName)]; }
    /**
     * Gets the well known name of this folder, if any.
     * **value** - The well known name of this folder, or null if this folder isn't a well known folder.
     *
     */
    get WellKnownFolderName() { return WellKnownFolderName_1.WellKnownFolderName[this.WellKnownFolderNameAsString] || null; }
    /**
     * _FolderTYpe -> type of folder, use to avoid folder type detection using instanceof. some cases it has circular loop in nodejs/requirejs
     */
    //get _FolderType(): string { return XmlElementNames.Folder; }
    /**
     * Initializes an unsaved local instance of **Folder**. To bind to an existing folder, use Folder.Bind() instead.
     *
     * @param   {ExchangeService}   service   EWS service to which this object belongs.
     */
    constructor(service) {
        super(service);
    }
    static Bind(service, idOrName, propertySet = PropertySet_1.PropertySet.FirstClassProperties) {
        if (idOrName instanceof FolderId_1.FolderId) {
            return service.BindToFolder(idOrName, propertySet);
        }
        else if (typeof idOrName === 'number') {
            return service.BindToFolder(new FolderId_1.FolderId(idOrName), propertySet);
        }
        EwsLogging_1.EwsLogging.Assert(false, "Folder.Bind", "unknown paramete type");
        throw new Error("unknow parameter type. this should not be  reached");
    }
    Copy(destinationFolderIdOrName) {
        this.ThrowIfThisIsNew();
        //EwsUtilities.ValidateParam(destinationFolderId, "destinationFolderId");
        if (typeof destinationFolderIdOrName === 'undefined') {
            EwsLogging_1.EwsLogging.Assert(false, "Folder.Copy", "unknown paramete type");
            throw new Error("unknow parameter type. this should not be  reached");
        }
        var folderId = destinationFolderIdOrName;
        if (typeof destinationFolderIdOrName === 'number')
            folderId = new FolderId_1.FolderId(destinationFolderIdOrName);
        return this.Service.CopyFolder(this.Id, folderId);
    }
    /**
     * Deletes the folder. Calling this method results in a call to EWS.
     *
     * @param   {DeleteMode}   deleteMode   Deletion mode.
     */
    Delete(deleteMode) { return this.InternalDelete(deleteMode, null, null); }
    /**
     * Empties the folder. Calling this method results in a call to EWS.
     *
     * @param   {DeleteMode}    deleteMode         The deletion mode.
     * @param   {boolean}       deleteSubFolders   Indicates whether sub-folders should also be deleted.
     */
    Empty(deleteMode, deleteSubFolders) {
        this.ThrowIfThisIsNew();
        return this.Service.EmptyFolder(this.Id, deleteMode, deleteSubFolders);
    }
    FindFolders(viewOrSearchFilter, view) {
        this.ThrowIfThisIsNew();
        //todo: better argument check with ewsutilities
        var argsLength = arguments.length;
        if (argsLength < 1 && argsLength > 2) {
            throw new Error("invalid arguments, check documentation and try again.");
        }
        if (viewOrSearchFilter instanceof FolderView_1.FolderView) {
            return this.Service.FindFolders(this.Id, viewOrSearchFilter);
        }
        else if (viewOrSearchFilter instanceof SearchFilter_1.SearchFilter) {
            if (typeof view === 'undefined' || !(view instanceof FolderView_1.FolderView)) {
                throw new Error("Folder.ts - FindFolders - incorrect uses of parameters at 2nd position, must be FolderView");
            }
            return this.Service.FindFolders(this.Id, viewOrSearchFilter, view);
        }
        else {
            throw new Error("Folder.ts - FindFolders - incorrect uses of parameters at 1st position, must be FolderView or SearchFilter");
        }
    }
    FindItems(viewQueryStringOrSearchFilter, viewOrGroupBy, groupBy) {
        var argsLength = arguments.length;
        if (argsLength < 1 && argsLength > 3) {
            throw new Error("invalid arguments, check documentation and try again.");
        }
        //todo: better argument check with ewsutilities
        //EwsUtilities.ValidateParam(groupBy, "groupBy");
        //EwsUtilities.ValidateParamAllowNull(searchFilter, "searchFilter");
        //EwsUtilities.ValidateParamAllowNull(queryString, "queryString");
        //position 1 - viewQueryStringOrSearchFilter
        var queryString = null;
        var searchFilter = null;
        var view = null;
        if (typeof viewQueryStringOrSearchFilter === 'string') {
            queryString = viewQueryStringOrSearchFilter;
        }
        else if (viewQueryStringOrSearchFilter instanceof SearchFilter_1.SearchFilter) {
            searchFilter = viewQueryStringOrSearchFilter;
        }
        else if (viewQueryStringOrSearchFilter instanceof ViewBase_1.ViewBase) {
            view = viewQueryStringOrSearchFilter;
        }
        else {
            throw new Error("Folder.ts - FindItems - incorrect uses of parameters at 1st position, must be string, Itemview or SearchFilter");
        }
        var groupResultBy = null;
        var isGroupped = false; // to resturn GroupedFindItemsResults<Item>
        //position 2 - viewOrGroupBy
        if (argsLength >= 2) {
            if (viewOrGroupBy instanceof Grouping_1.Grouping) {
                if (!(viewQueryStringOrSearchFilter instanceof ItemView_1.ItemView)) {
                    throw new Error("Folder.ts - FindItems with " + argsLength + " parameters - incorrect uses of parameter at 1nd position, it must be Itemview when using Grouping at 2nd place");
                }
                groupResultBy = viewOrGroupBy;
                isGroupped = true;
            }
            else if (viewOrGroupBy instanceof ItemView_1.ItemView) {
                view = viewOrGroupBy;
            }
            else {
                throw new Error("ExchangeService.ts - FindItems with " + argsLength + " parameters - incorrect uses of parameter at 2nd position, must be Itemsview or Grouping");
            }
        }
        //position 3 - groupBy
        if (argsLength === 3) {
            if (!(viewOrGroupBy instanceof ItemView_1.ItemView)) {
                throw new Error("Folder.ts - FindItems with " + argsLength + " parameters - incorrect uses of parameter at 1nd position, it must be Itemview when using Grouping at 3rd place");
            }
            groupResultBy = groupBy;
            isGroupped = true;
        }
        return this.InternalFindItems(searchFilter || queryString, view, groupResultBy /* groupBy */)
            .then((res) => {
            if (isGroupped) {
                return res.__thisIndexer(0).GroupedFindResults;
            }
            return res.__thisIndexer(0).Results;
        });
    }
    /**
     * @internal Gets the element name of item in XML
     *
     * @return  {string} name of elelment
     */
    GetXmlElementName() { return XmlElementNames_1.XmlElementNames.Folder; }
    /**
     * @internal Gets the name of the change XML element.
     *
     * @return  {string}      XML element name,
     */
    GetChangeXmlElementName() { return XmlElementNames_1.XmlElementNames.FolderChange; }
    /**
     * @internal Gets the name of the delete field XML element.
     *
     * @return  {string}      XML element name,
     */
    GetDeleteFieldXmlElementName() { return XmlElementNames_1.XmlElementNames.DeleteFolderField; }
    /**
     * @internal Gets a list of extended properties defined on this object.
     *
     * @return  {ExtendedPropertyCollection}      Extended properties collection.
     */
    GetExtendedProperties() { return this.ExtendedProperties; }
    /**
     * @internal Get the property definition for the Id property.
     *
     * @return  {PropertyDefinition}      A PropertyDefinition instance.
     */
    GetIdPropertyDefinition() { return Schemas_1.Schemas.FolderSchema.Id; }
    /**
     * @internal Gets the minimum required server version.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this service object type is supported.
     */
    GetMinimumRequiredServerVersion() { return ExchangeVersion_1.ExchangeVersion.Exchange2007_SP1; }
    /**
     * @internal Internal method to return the schema associated with this type of object.
     *
     * @return  {ServiceObjectSchema}      The schema associated with this type of object.
     */
    GetSchema() { return Schemas_1.Schemas.FolderSchema.Instance; }
    /**
     * @internal Gets the name of the set field XML element.
     *
     * @return  {string}      XML element name,
     */
    GetSetFieldXmlElementName() { return XmlElementNames_1.XmlElementNames.SetFolderField; }
    /**
     * @internal Deletes the object.
     *
     * @param   {DeleteMode}                deleteMode                The deletion mode.
     * @param   {SendCancellationsMode}     sendCancellationsMode     Indicates whether meeting cancellation messages should be sent.
     * @param   {AffectedTaskOccurrence}    affectedTaskOccurrences   Indicate which occurrence of a recurring task should be deleted.
     */
    InternalDelete(deleteMode, sendCancellationsMode, affectedTaskOccurrences) {
        this.ThrowIfThisIsNew();
        return this.Service.DeleteFolder(this.Id, deleteMode);
    }
    InternalFindItems(searchFilterOrQueryString, view, groupBy) {
        this.ThrowIfThisIsNew();
        var searchFilter = null;
        var queryString = null;
        if (searchFilterOrQueryString instanceof SearchFilter_1.SearchFilter) {
            searchFilter = searchFilterOrQueryString;
        }
        else if (typeof searchFilterOrQueryString === 'string') {
            queryString = searchFilterOrQueryString;
        }
        //debug: //todo: //ref: verify if querystring is null
        return this.Service.FindItems([this.Id], // FolderId[]
        searchFilter, /* searchFilter */ queryString, /* queryString */ view, groupBy, ServiceErrorHandling_1.ServiceErrorHandling.ThrowOnError);
    }
    /**
     * @internal Loads the specified set of properties on the object.
     *
     * @param   {PropertySet}   propertySet   The properties to load.
     */
    InternalLoad(propertySet) {
        this.ThrowIfThisIsNew();
        return this.Service.LoadPropertiesForFolder(this, propertySet);
    }
    /**
     * Marks all items in folder as read. Calling this method results in a call to EWS.
     *
     * @param   {boolean}   suppressReadReceipts   If true, suppress sending read receipts for items.
     */
    MarkAllItemsAsRead(suppressReadReceipts) {
        this.ThrowIfThisIsNew();
        return this.Service.MarkAllItemsAsRead(this.Id, true, suppressReadReceipts);
    }
    /**
     * Marks all items in folder as read. Calling this method results in a call to EWS.
     *
     * @param   {boolean}   suppressReadReceipts   If true, suppress sending read receipts for items.
     */
    MarkAllItemsAsUnread(suppressReadReceipts) {
        this.ThrowIfThisIsNew();
        return this.Service.MarkAllItemsAsRead(this.Id, false, suppressReadReceipts);
    }
    Move(destinationFolderIdOrName) {
        this.ThrowIfThisIsNew();
        if (typeof destinationFolderIdOrName === 'undefined') {
            EwsLogging_1.EwsLogging.Assert(false, "Folder.Move", "unknown paramete type");
            throw new Error("unknow parameter type. this should not be  reached");
        }
        //EwsUtilities.ValidateParam(destinationFolderId, "destinationFolderId");
        var folderId = destinationFolderIdOrName;
        if (typeof destinationFolderIdOrName === 'number')
            folderId = new FolderId_1.FolderId(destinationFolderIdOrName);
        return this.Service.MoveFolder(this.Id, folderId);
    }
    /**
     * Removes an extended property.
     *
     * @param   {ExtendedPropertyDefinition}   extendedPropertyDefinition   The extended property definition.
     * @return  {boolean}       True if property was removed.
     */
    RemoveExtendedProperty(extendedPropertyDefinition) { return this.ExtendedProperties.RemoveExtendedProperty(extendedPropertyDefinition); }
    Save(parentFolderIdOrname) {
        this.ThrowIfThisIsNotNew();
        if (typeof parentFolderIdOrname === 'undefined') {
            EwsLogging_1.EwsLogging.Assert(false, "Folder.Save", "unknown paramete type");
            throw new Error("unknow parameter type. this should not be  reached");
        }
        //EwsUtilities.ValidateParam(parentFolderId, "parentFolderId");
        var folderId = parentFolderIdOrname;
        if (typeof parentFolderIdOrname === 'number')
            folderId = new FolderId_1.FolderId(parentFolderIdOrname);
        if (this.IsDirty) {
            return this.Service.CreateFolder(this, folderId);
        }
        else
            return null;
    }
    /**
     * Sets the extended property.
     *
     * @param   {ExtendedPropertyDefinition}    extendedPropertyDefinition   The extended property definition.
     * @param   {any}                           value                        The value.
     */
    SetExtendedProperty(extendedPropertyDefinition, value) { this.ExtendedProperties.SetExtendedProperty(extendedPropertyDefinition, value); }
    /**
     * Applies the local changes that have been made to this folder. Calling this method results in a call to EWS.
     *
     */
    Update() {
        if (this.IsDirty) {
            if (this.PropertyBag.GetIsUpdateCallNecessary()) {
                return this.Service.UpdateFolder(this);
            }
        }
        return undefined;
    }
    /**
     * @internal Validates this instance.
     *
     */
    Validate() {
        super.Validate();
        // Validate folder permissions
        if (this.PropertyBag.Contains(Schemas_1.Schemas.FolderSchema.Permissions)) {
            this.Permissions.Validate();
        }
    }
}
exports.Folder = Folder;
//# sourceMappingURL=Folder.js.map