"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmailMessage = void 0;
const ConflictResolutionMode_1 = require("../../../Enumerations/ConflictResolutionMode");
const ExchangeVersion_1 = require("../../../Enumerations/ExchangeVersion");
const FolderId_1 = require("../../../ComplexProperties/FolderId");
const MessageDisposition_1 = require("../../../Enumerations/MessageDisposition");
const PropertySet_1 = require("../../PropertySet");
const ResponseMessage_1 = require("../ResponseObjects/ResponseMessage");
const ResponseMessageType_1 = require("../../../Enumerations/ResponseMessageType");
const Schemas_1 = require("../Schemas/Schemas");
const SuppressReadReceipt_1 = require("../ResponseObjects/SuppressReadReceipt");
const WellKnownFolderName_1 = require("../../../Enumerations/WellKnownFolderName");
const XmlElementNames_1 = require("../../XmlElementNames");
const Item_1 = require("./Item");
/**
 * Represents an **e-mail message**. Properties available on e-mail messages are defined in the *EmailMessageSchema* class.
 *
 */
class EmailMessage extends Item_1.Item {
    /** required to check [Attachable] attribute, AttachmentCollection.AddItemAttachment<TItem>() checks for non inherited [Attachable] attribute. */
    static get Attachable() { return this.name === "EmailMessage"; }
    ;
    /**
     * Gets the list of To recipients for the e-mail message.
     *
     */
    get ToRecipients() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.ToRecipients);
    }
    /**
     * Gets the list of Bcc recipients for the e-mail message.
     *
     */
    get BccRecipients() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.BccRecipients);
    }
    /**
     * Gets the list of Cc recipients for the e-mail message.
     *
     */
    get CcRecipients() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.CcRecipients);
    }
    /**
     * Gets the conversation topic of the e-mail message.
     *
     */
    get ConversationTopic() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.ConversationTopic);
    }
    /**
     * Gets the conversation index of the e-mail message.
     *
     */
    get ConversationIndex() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.ConversationIndex);
    }
    /**
     * Gets or sets the "on behalf" sender of the e-mail message.
     *
     */
    get From() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.From);
    }
    set From(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.EmailMessageSchema.From, value);
    }
    /**
     * Gets or sets a value indicating whether this is an associated message.
     *
     */
    get IsAssociated() {
        return this.IsAssociated;
    }
    set IsAssociated(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.IsAssociated, value);
    }
    /**
     * Gets or sets a value indicating whether a read receipt is requested for the e-mail message.
     *
     */
    get IsDeliveryReceiptRequested() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.IsDeliveryReceiptRequested);
    }
    set IsDeliveryReceiptRequested(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.EmailMessageSchema.IsDeliveryReceiptRequested, value);
    }
    /**
     * Gets or sets a value indicating whether the e-mail message is read.
     *
     */
    get IsRead() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.IsRead);
    }
    set IsRead(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.EmailMessageSchema.IsRead, value);
    }
    /**
     * Gets or sets a value indicating whether a read receipt is requested for the e-mail message.
     *
     */
    get IsReadReceiptRequested() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.IsReadReceiptRequested);
    }
    set IsReadReceiptRequested(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.EmailMessageSchema.IsReadReceiptRequested, value);
    }
    /**
     * Gets or sets a value indicating whether a response is requested for the e-mail message.
     *
     */
    get IsResponseRequested() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.IsResponseRequested);
    }
    set IsResponseRequested(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.EmailMessageSchema.IsResponseRequested, value);
    }
    /**
     * Gets the Internat Message Id of the e-mail message.
     *
     */
    get InternetMessageId() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.InternetMessageId);
    }
    /**
     * Gets or sets the references of the e-mail message.
     *
     */
    get References() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.References);
    }
    set References(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.EmailMessageSchema.References, value);
    }
    /**
     * Gets a list of e-mail addresses to which replies should be addressed.
     *
     */
    get ReplyTo() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.ReplyTo);
    }
    /**
     * Gets or sets the sender of the e-mail message.
     *
     */
    get Sender() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.Sender);
    }
    set Sender(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.EmailMessageSchema.Sender, value);
    }
    /**
     * Gets the ReceivedBy property of the e-mail message.
     *
     */
    get ReceivedBy() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.ReceivedBy);
    }
    /**
     * Gets the ReceivedRepresenting property of the e-mail message.
     *
     */
    get ReceivedRepresenting() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.ReceivedRepresenting);
    }
    /**
     * Gets the ApprovalRequestData property of the e-mail message.
     *
     */
    get ApprovalRequestData() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.ApprovalRequestData);
    }
    /**
     * Gets the VotingInformation property of the e-mail message.
     *
     */
    get VotingInformation() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.VotingInformation);
    }
    constructor(serviceOrParentAttachment) {
        super(serviceOrParentAttachment);
    }
    static Bind(service, id, propertySet = PropertySet_1.PropertySet.FirstClassProperties) {
        return service.BindToItem(id, propertySet, EmailMessage);
    }
    /**
     * Creates a forward response to the message.
     *
     * @return  {ResponseMessage}      A ResponseMessage representing the forward response that can subsequently be modified and sent.
     */
    CreateForward() {
        this.ThrowIfThisIsNew();
        return new ResponseMessage_1.ResponseMessage(this, ResponseMessageType_1.ResponseMessageType.Forward);
    }
    /**
     * Creates a reply response to the message.
     *
     * @param   {boolean}             replyAll   Indicates whether the reply should go to all of the original recipients of the message.
     * @return  {ResponseMessage}     A ResponseMessage representing the reply response that can subsequently be modified and sent.
     */
    CreateReply(replyAll) {
        this.ThrowIfThisIsNew();
        return new ResponseMessage_1.ResponseMessage(this, replyAll ? ResponseMessageType_1.ResponseMessageType.ReplyAll : ResponseMessageType_1.ResponseMessageType.Reply);
    }
    //Forward(bodyPrefix: MessageBody, toRecipients: EmailAddress[]): Promise<void> { throw new Error("EmailMessage.ts - Forward : Not implemented."); }
    //Forward(bodyPrefix: MessageBody, toRecipients: System.Collections.Generic.IEnumerable<T>): Promise<void> { throw new Error("EmailMessage.ts - Forward : Not implemented."); }
    /**
     * Forwards the message. Calling this method results in a call to EWS.
     *
     * @param   {MessageBody}   bodyPrefix     The prefix to prepend to the original body of the message.
     * @param   {EmailAddress[]}   toRecipients   The recipients to forward the message to.
     */
    Forward(bodyPrefix, toRecipients) {
        var responseMessage = this.CreateForward();
        responseMessage.BodyPrefix = bodyPrefix;
        responseMessage.ToRecipients.AddRange(toRecipients);
        return responseMessage.SendAndSaveCopy();
    }
    /**
     * @internal Gets the minimum required server version.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this service object type is supported.
     */
    GetMinimumRequiredServerVersion() { return ExchangeVersion_1.ExchangeVersion.Exchange2007_SP1; }
    /**
     * @internal Internal method to return the schema associated with this type of object.
     *
     * @return  {ServiceObjectSchema}      The schema associated with this type of object.
     */
    GetSchema() { return Schemas_1.Schemas.EmailMessageSchema.Instance; }
    /**
     * @internal Gets the element name of item in XML
     *
     * @return  {string} name of elelment
     */
    GetXmlElementName() { return XmlElementNames_1.XmlElementNames.Message; }
    /**
     * Send message.
     *
     * @param   {FolderId}            parentFolderId       The parent folder id.
     * @param   {MessageDisposition}  messageDisposition   The message disposition.
     */
    InternalSend(parentFolderId, messageDisposition) {
        this.ThrowIfThisIsAttachment();
        if (this.IsNew) {
            if ((this.Attachments.Count == 0) || (messageDisposition == MessageDisposition_1.MessageDisposition.SaveOnly)) {
                return this.InternalCreate(parentFolderId, messageDisposition, null);
            }
            else {
                // If the message has attachments, save as a draft (and add attachments) before sending.
                return this.InternalCreate(null, // null means use the Drafts folder in the mailbox of the authenticated user.
                MessageDisposition_1.MessageDisposition.SaveOnly, null).then(() => {
                    return this.Service.SendItem(this, parentFolderId);
                });
            }
        }
        else {
            // Regardless of whether item is dirty or not, if it has unprocessed
            // attachment changes, process them now.
            //debugger; //todo: check - check for attachment save() promise. 
            return Promise.resolve(
            // Validate and save attachments before sending.
            this.HasUnprocessedAttachmentChanges() ? this.Attachments.ValidateAndSave() : void 0)
                .then(() => {
                if (this.PropertyBag.GetIsUpdateCallNecessary()) {
                    return this.InternalUpdate(//ref: //info: <any> to supress cast error, returning promise is required, this time it is not void but no action is taken on this promise. 
                    parentFolderId, ConflictResolutionMode_1.ConflictResolutionMode.AutoResolve, messageDisposition, null);
                }
                else {
                    return this.Service.SendItem(this, parentFolderId);
                }
            });
        }
    }
    /**
     * Replies to the message. Calling this method results in a call to EWS.
     *
     * @param   {MessageBody}   bodyPrefix   The prefix to prepend to the original body of the message.
     * @param   {boolean}   replyAll     Indicates whether the reply should be sent to all of the original recipients of the message.
     */
    Reply(bodyPrefix, replyAll) {
        var responseMessage = this.CreateReply(replyAll);
        responseMessage.BodyPrefix = bodyPrefix;
        return responseMessage.SendAndSaveCopy();
    }
    /**
     * Sends this e-mail message. Calling this method results in at least one call to EWS.
     */
    Send() { return this.InternalSend(null, MessageDisposition_1.MessageDisposition.SendOnly); }
    SendAndSaveCopy(destinationFolderIdOrName) {
        var destinationFolderId = new FolderId_1.FolderId(WellKnownFolderName_1.WellKnownFolderName.SentItems);
        if (arguments.length === 1) {
            if (typeof destinationFolderIdOrName === "number") {
                destinationFolderId = new FolderId_1.FolderId(destinationFolderIdOrName);
            }
            else {
                //EwsUtilities.ValidateParam(destinationFolderIdOrName, "destinationFolderId");
                destinationFolderId = destinationFolderIdOrName;
            }
        }
        return this.InternalSend(destinationFolderId, MessageDisposition_1.MessageDisposition.SendAndSaveCopy);
    }
    /**
     * Suppresses the read receipt on the message. Calling this method results in a call to EWS.
     *
     */
    SuppressReadReceipt() {
        this.ThrowIfThisIsNew();
        return (new SuppressReadReceipt_1.SuppressReadReceipt(this)).InternalCreate(null, null);
    }
}
exports.EmailMessage = EmailMessage;
//# sourceMappingURL=EmailMessage.js.map