"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PostReply = void 0;
const EwsLogging_1 = require("../../EwsLogging");
const EwsUtilities_1 = require("../../EwsUtilities");
const ExchangeVersion_1 = require("../../../Enumerations/ExchangeVersion");
const FolderId_1 = require("../../../ComplexProperties/FolderId");
const InvalidOperationException_1 = require("../../../Exceptions/InvalidOperationException");
const PostItem_1 = require("../Items/PostItem");
const Schemas_1 = require("../Schemas/Schemas");
const Strings_1 = require("../../../Strings");
const XmlElementNames_1 = require("../../XmlElementNames");
const ServiceObject_1 = require("../ServiceObject");
/**
 * Represents a reply to a post item.
 *
 * @sealed
 */
class PostReply extends ServiceObject_1.ServiceObject {
    /**
     * @internal Initializes a new instance of the **PostReply** class.
     *
     * @param   {Item}   referenceItem   The reference item.
     */
    constructor(referenceItem) {
        super(referenceItem.Service);
        this.referenceItem = null;
        EwsLogging_1.EwsLogging.Assert(referenceItem != null, "PostReply.ctor", "referenceItem is null");
        referenceItem.ThrowIfThisIsNew();
        this.referenceItem = referenceItem;
    }
    /**
     * Gets or sets the subject of the post reply.
     */
    get Subject() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.Subject);
    }
    set Subject(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.Subject, value);
    }
    /**
     * Gets or sets the body of the post reply.
     */
    get Body() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.Body);
    }
    set Body(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.Body, value);
    }
    /**
     * Gets or sets the body prefix that should be prepended to the original post item's body.
     */
    get BodyPrefix() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ResponseObjectSchema.BodyPrefix);
    }
    set BodyPrefix(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ResponseObjectSchema.BodyPrefix, value);
    }
    /**
     * @internal Gets the minimum required server version.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this service object type is supported.
     */
    GetMinimumRequiredServerVersion() {
        return ExchangeVersion_1.ExchangeVersion.Exchange2007_SP1;
    }
    /**
     * @internal Internal method to return the schema associated with this type of object.
     *
     * @return  {ServiceObjectSchema}      The schema associated with this type of object.
     */
    GetSchema() {
        return Schemas_1.Schemas.PostReplySchema.Instance;
    }
    /**
     * @internal Gets the element name of item in XML
     *
     * @return  {string} name of elelment
     */
    GetXmlElementName() {
        return XmlElementNames_1.XmlElementNames.PostReplyItem;
    }
    /**
     * @internal Create a PostItem response.
     *
     * @param   {FolderId}              parentFolderId       The parent folder id.
     * @param   {MessageDisposition}    messageDisposition   The message disposition.
     * @return  {Promise<PostItem>}    Created PostItem    :Promise.
     */
    InternalCreate(parentFolderId, messageDisposition) {
        this.PropertyBag._getItem(Schemas_1.Schemas.ResponseObjectSchema.ReferenceItemId).Assign(this.referenceItem.Id);
        return this.Service.InternalCreateResponseObject(this, parentFolderId, messageDisposition).then((items) => {
            let postItem = EwsUtilities_1.EwsUtilities.FindFirstItemOfType(items, PostItem_1.PostItem);
            // This should never happen. If it does, we have a bug.
            EwsLogging_1.EwsLogging.Assert(postItem != null, "PostReply.InternalCreate", "postItem is null. The CreateItem call did not return the expected PostItem.");
            return postItem;
        });
    }
    /**
     * @internal Deletes the object.
     *
     * @param   {DeleteMode}                deleteMode                The deletion mode.
     * @param   {SendCancellationsMode}     sendCancellationsMode     Indicates whether meeting cancellation messages should be sent.
     * @param   {AffectedTaskOccurrence}    affectedTaskOccurrences   Indicate which occurrence of a recurring task should be deleted.
     * @return  {Promise<void>}    :Promise.
     */
    InternalDelete(deleteMode, sendCancellationsMode, affectedTaskOccurrences) {
        throw new InvalidOperationException_1.InvalidOperationException(Strings_1.Strings.DeletingThisObjectTypeNotAuthorized);
    }
    /**
     * @internal Loads the specified set of properties on the object.
     *
     * @param   {PropertySet}   propertySet   The properties to load.
     * @return  {Promise<void>}    :Promise.
     */
    InternalLoad(propertySet) {
        throw new InvalidOperationException_1.InvalidOperationException(Strings_1.Strings.LoadingThisObjectTypeNotSupported);
    }
    Save(destinationFolderIdOrFolderName = null) {
        let destinationFolderId = destinationFolderIdOrFolderName;
        if (arguments.length > 0) {
            EwsUtilities_1.EwsUtilities.ValidateParam(destinationFolderIdOrFolderName, "destinationFolderId");
        }
        if (typeof destinationFolderIdOrFolderName === 'number') {
            destinationFolderId = new FolderId_1.FolderId(destinationFolderIdOrFolderName);
        }
        return this.InternalCreate(destinationFolderId, null);
    }
}
exports.PostReply = PostReply;
//# sourceMappingURL=PostReply.js.map