"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResponseMessage = void 0;
const Schemas_1 = require("../Schemas/Schemas");
const XmlElementNames_1 = require("../../XmlElementNames");
const EwsLogging_1 = require("../../EwsLogging");
const ResponseMessageType_1 = require("../../../Enumerations/ResponseMessageType");
const ExchangeVersion_1 = require("../../../Enumerations/ExchangeVersion");
const ResponseObject_1 = require("./ResponseObject");
/**
 * Represents the base class for e-mail related responses (Reply, Reply all and Forward).
 *
 */
class ResponseMessage extends ResponseObject_1.ResponseObject {
    /**
     * Initializes a new instance of the **ResponseMessage** class.
     *
     * @param   {Item}                    referenceItem   The reference item.
     * @param   {ResponseMessageType}     responseType    Type of the response.
     */
    constructor(referenceItem, responseType) {
        super(referenceItem);
        this.responseType = ResponseMessageType_1.ResponseMessageType.Reply;
        this.responseType = responseType;
    }
    /**
     * Gets a value indicating the type of response this object represents.
     *
     */
    get ResponseType() {
        return this.responseType;
    }
    /**
     * Gets or sets the body of the response.
     *
     */
    get Body() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.Body);
    }
    set Body(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.Body, value);
    }
    /**
     * Gets a list of recipients the response will be sent to.
     *
     */
    get ToRecipients() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.ToRecipients);
    }
    /**
     * Gets a list of recipients the response will be sent to as Cc.
     *
     */
    get CcRecipients() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.CcRecipients);
    }
    /**
     * Gets a list of recipients this response will be sent to as Bcc.
     *
     */
    get BccRecipients() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.BccRecipients);
    }
    /**
     * Gets or sets the subject of this response.
     *
     */
    get Subject() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.Subject);
    }
    set Subject(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.Subject, value);
    }
    /**
     * Gets or sets the body prefix of this response. The body prefix will be prepended to the original
    message's body when the response is created.
     *
     */
    get BodyPrefix() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ResponseObjectSchema.BodyPrefix);
    }
    set BodyPrefix(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ResponseObjectSchema.BodyPrefix, value);
    }
    /**
     * Gets the minimum required server version.
     *
     * @return  {type}      Earliest Exchange version in which this service object type is supported.
     */
    GetMinimumRequiredServerVersion() { return ExchangeVersion_1.ExchangeVersion.Exchange2007_SP1; }
    /**
     * Internal method to return the schema associated with this type of object.
     *
     * @return  {ServiceObjectSchema}      The schema associated with this type of object.
     */
    GetSchema() { return Schemas_1.Schemas.ResponseMessageSchema.Instance; }
    /**
     * Get XML Element Name - workaround for c# attributes
     */
    GetXmlElementName() { return this.GetXmlElementNameOverride(); }
    /**
     * This methods lets subclasses of ServiceObject override the default mechanism by which the XML element name associated with their type is retrieved.
     *
     * @return  {string}      The XML element name associated with this type. If this method returns null or empty, the XML element name associated with this type is determined by the EwsObjectDefinition attribute that decorates the type, if present.
     */
    GetXmlElementNameOverride() {
        switch (this.responseType) {
            case ResponseMessageType_1.ResponseMessageType.Reply:
                return XmlElementNames_1.XmlElementNames.ReplyToItem;
            case ResponseMessageType_1.ResponseMessageType.ReplyAll:
                return XmlElementNames_1.XmlElementNames.ReplyAllToItem;
            case ResponseMessageType_1.ResponseMessageType.Forward:
                return XmlElementNames_1.XmlElementNames.ForwardItem;
            default:
                EwsLogging_1.EwsLogging.Assert(false, "ResponseMessage.GetXmlElementNameOverride", "An unexpected value for responseType could not be handled.");
                return null; // Because the compiler wants it
        }
    }
}
exports.ResponseMessage = ResponseMessage;
//# sourceMappingURL=ResponseMessage.js.map