"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SearchMailboxesResult = void 0;
const ExtensionMethods_1 = require("../ExtensionMethods");
const DateTime_1 = require("../DateTime");
const EwsServiceJsonReader_1 = require("../Core/EwsServiceJsonReader");
const ExtendedPropertyCollection_1 = require("../ComplexProperties/ExtendedPropertyCollection");
const FailedSearchMailbox_1 = require("./FailedSearchMailbox");
const Importance_1 = require("../Enumerations/Importance");
const ItemId_1 = require("../ComplexProperties/ItemId");
const KeywordStatisticsSearchResult_1 = require("./KeywordStatisticsSearchResult");
const MailboxQuery_1 = require("./MailboxQuery");
const MailboxSearchLocation_1 = require("../Enumerations/MailboxSearchLocation");
const MailboxSearchScope_1 = require("./MailboxSearchScope");
const MailboxStatisticsItem_1 = require("./MailboxStatisticsItem");
const PreviewItemMailbox_1 = require("./PreviewItemMailbox");
const SearchPreviewItem_1 = require("./SearchPreviewItem");
const SearchRefinerItem_1 = require("./SearchRefinerItem");
const SearchResultType_1 = require("../Enumerations/SearchResultType");
const XmlElementNames_1 = require("../Core/XmlElementNames");
/**
 * Represents search mailbox result.
 *
 * @sealed
 */
class SearchMailboxesResult {
    constructor() {
        /**
         * Search queries
         */
        this.SearchQueries = null;
        /**
         * Result type
         */
        this.ResultType = SearchResultType_1.SearchResultType.StatisticsOnly;
        /**
         * Item count
         */
        this.ItemCount = 0;
        /**
         * Total size
         * [CLSCompliant(false)]
         */
        this.Size = 0;
        /**
         * Page item count
         */
        this.PageItemCount = 0;
        /**
         * Total page item size
         * [CLSCompliant(false)]
         */
        this.PageItemSize = 0;
        /**
         * Keyword statistics search result
         */
        this.KeywordStats = null;
        /**
         * Search preview items
         */
        this.PreviewItems = null;
        /**
         * Failed mailboxes
         */
        this.FailedMailboxes = null;
        /**
         * Refiners
         */
        this.Refiners = null;
        /**
         * Mailbox statistics
         */
        this.MailboxStats = null;
    }
    /**
     * Get collection of recipients
     *
     * @param   {any}   jsObject	Json Object converted from XML.
     * @return  {string[]}          Array of recipients
     */
    static GetRecipients(jsObject) {
        let recipients = EwsServiceJsonReader_1.EwsServiceJsonReader.ReadAsArray(jsObject, XmlElementNames_1.XmlElementNames.SmtpAddress);
        return recipients.length === 0 ? null : recipients;
    }
    /**
     * Load extended properties from XML.
     *
     * @param   {any}				jsObject	Json Object converted from XML.
     * @param   {ExchangeService}	service	The service.
     * @return  {ExtendedPropertyCollection}     Extended properties collection
     */
    static LoadExtendedPropertiesXmlJsObject(jsObject, service) {
        let extendedProperties = new ExtendedPropertyCollection_1.ExtendedPropertyCollection();
        for (let extendedProperty of EwsServiceJsonReader_1.EwsServiceJsonReader.ReadAsArray(jsObject, XmlElementNames_1.XmlElementNames.ExtendedProperty)) {
            extendedProperties.LoadFromXmlJsObject(extendedProperty, service);
        }
        return extendedProperties.Count === 0 ? null : extendedProperties;
    }
    /**
     * Loads service object from XML.
     *
     * @param   {any}				jsObject	Json Object converted from XML.
     * @param   {ExchangeService}	service	The service.
     * @return  {SearchMailboxesResult}     Search result object
     */
    static LoadFromXmlJsObject(jsObject, service) {
        let searchResult = new SearchMailboxesResult();
        if (jsObject[XmlElementNames_1.XmlElementNames.SearchQueries]) {
            searchResult.SearchQueries = [];
            for (let searchQuery of EwsServiceJsonReader_1.EwsServiceJsonReader.ReadAsArray(jsObject[XmlElementNames_1.XmlElementNames.SearchQueries], XmlElementNames_1.XmlElementNames.SearchQuery)) {
                let query = searchQuery[XmlElementNames_1.XmlElementNames.Query];
                let mailboxSearchScopes = [];
                if (searchQuery[XmlElementNames_1.XmlElementNames.MailboxSearchScopes]) {
                    for (let mailboxSearchScope of EwsServiceJsonReader_1.EwsServiceJsonReader.ReadAsArray(searchQuery[XmlElementNames_1.XmlElementNames.MailboxSearchScopes], XmlElementNames_1.XmlElementNames.MailboxSearchScope)) {
                        let mailbox = mailboxSearchScope[XmlElementNames_1.XmlElementNames.Mailbox];
                        let searchScope = MailboxSearchLocation_1.MailboxSearchLocation[mailboxSearchScope[XmlElementNames_1.XmlElementNames.SearchScope]];
                        mailboxSearchScopes.push(new MailboxSearchScope_1.MailboxSearchScope(mailbox, searchScope));
                    }
                }
                searchResult.SearchQueries.push(new MailboxQuery_1.MailboxQuery(query, mailboxSearchScopes));
            }
        }
        if (jsObject[XmlElementNames_1.XmlElementNames.ResultType]) {
            searchResult.ResultType = SearchResultType_1.SearchResultType[jsObject[XmlElementNames_1.XmlElementNames.ResultType]];
        }
        if (jsObject[XmlElementNames_1.XmlElementNames.ItemCount]) {
            searchResult.ItemCount = ExtensionMethods_1.Convert.toNumber(jsObject[XmlElementNames_1.XmlElementNames.ItemCount]);
        }
        if (jsObject[XmlElementNames_1.XmlElementNames.Size]) {
            searchResult.Size = ExtensionMethods_1.Convert.toNumber(jsObject[XmlElementNames_1.XmlElementNames.Size]);
        }
        if (jsObject[XmlElementNames_1.XmlElementNames.PageItemCount]) {
            searchResult.PageItemCount = ExtensionMethods_1.Convert.toNumber(jsObject[XmlElementNames_1.XmlElementNames.PageItemCount]);
        }
        if (jsObject[XmlElementNames_1.XmlElementNames.PageItemSize]) {
            searchResult.PageItemSize = ExtensionMethods_1.Convert.toNumber(jsObject[XmlElementNames_1.XmlElementNames.PageItemSize]);
        }
        if (jsObject[XmlElementNames_1.XmlElementNames.KeywordStats]) {
            searchResult.KeywordStats = this.LoadKeywordStatsXmlJsObject(jsObject[XmlElementNames_1.XmlElementNames.KeywordStats]);
        }
        if (jsObject[XmlElementNames_1.XmlElementNames.Items]) {
            searchResult.PreviewItems = this.LoadPreviewItemsXmlJsObject(jsObject[XmlElementNames_1.XmlElementNames.Items], service);
        }
        if (jsObject[XmlElementNames_1.XmlElementNames.FailedMailboxes]) {
            searchResult.FailedMailboxes = FailedSearchMailbox_1.FailedSearchMailbox.LoadFromXmlJsObject(jsObject[XmlElementNames_1.XmlElementNames.FailedMailboxes], service);
        }
        if (jsObject[XmlElementNames_1.XmlElementNames.Refiners]) {
            let refiners = [];
            for (let refiner of EwsServiceJsonReader_1.EwsServiceJsonReader.ReadAsArray(jsObject[XmlElementNames_1.XmlElementNames.Refiners], XmlElementNames_1.XmlElementNames.Refiner)) {
                refiners.push(SearchRefinerItem_1.SearchRefinerItem.LoadFromXmlJsObject(refiner, service));
            }
            if (refiners.length > 0) {
                searchResult.Refiners = refiners;
            }
        }
        if (jsObject[XmlElementNames_1.XmlElementNames.MailboxStats]) {
            let mailboxStats = [];
            for (let mailboxStat of EwsServiceJsonReader_1.EwsServiceJsonReader.ReadAsArray(jsObject[XmlElementNames_1.XmlElementNames.MailboxStats], XmlElementNames_1.XmlElementNames.MailboxStat)) {
                mailboxStats.push(MailboxStatisticsItem_1.MailboxStatisticsItem.LoadFromXmlJsObject(mailboxStat, service));
            }
            if (mailboxStats.length > 0) {
                searchResult.MailboxStats = mailboxStats;
            }
        }
        return searchResult;
    }
    /**
     * Load keyword stats from XML.
     *
     * @param   {any}   jsObject	Json Object converted from XML.
     * @return  {KeywordStatisticsSearchResult[]}       Array of keyword statistics
     */
    static LoadKeywordStatsXmlJsObject(jsObject) {
        let keywordStats = [];
        for (let keywordStatObj of EwsServiceJsonReader_1.EwsServiceJsonReader.ReadAsArray(jsObject, XmlElementNames_1.XmlElementNames.KeywordStat)) {
            let keywordStat = new KeywordStatisticsSearchResult_1.KeywordStatisticsSearchResult();
            keywordStat.Keyword = jsObject[XmlElementNames_1.XmlElementNames.Keyword];
            keywordStat.ItemHits = ExtensionMethods_1.Convert.toNumber(jsObject[XmlElementNames_1.XmlElementNames.ItemHits]);
            keywordStat.Size = ExtensionMethods_1.Convert.toNumber(jsObject[XmlElementNames_1.XmlElementNames.Size]);
            keywordStats.push(keywordStat);
        }
        return keywordStats.length === 0 ? null : keywordStats;
    }
    /**
     * Load preview items from XML.
     *
     * @param   {any}				jsObject	Json Object converted from XML.
     * @param   {ExchangeService}	service	The service.
     * @return  {SearchPreviewItem[]}     Array of preview items
     */
    static LoadPreviewItemsXmlJsObject(jsObject, service) {
        let previewItems = [];
        for (let searchPreviewItem of EwsServiceJsonReader_1.EwsServiceJsonReader.ReadAsArray(jsObject, XmlElementNames_1.XmlElementNames.SearchPreviewItem)) {
            let previewItem = new SearchPreviewItem_1.SearchPreviewItem();
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.Id]) {
                previewItem.Id = new ItemId_1.ItemId();
                previewItem.Id.LoadFromXmlJsObject(searchPreviewItem[XmlElementNames_1.XmlElementNames.Id], service);
            }
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.ParentId]) {
                previewItem.ParentId = new ItemId_1.ItemId();
                previewItem.ParentId.LoadFromXmlJsObject(searchPreviewItem[XmlElementNames_1.XmlElementNames.ParentId], service);
            }
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.Mailbox]) {
                previewItem.Mailbox = new PreviewItemMailbox_1.PreviewItemMailbox();
                previewItem.Mailbox.MailboxId = searchPreviewItem[XmlElementNames_1.XmlElementNames.Mailbox][XmlElementNames_1.XmlElementNames.MailboxId];
                previewItem.Mailbox.PrimarySmtpAddress = searchPreviewItem[XmlElementNames_1.XmlElementNames.Mailbox][XmlElementNames_1.XmlElementNames.PrimarySmtpAddress];
            }
            //missing in official repo
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.ItemClass]) {
                previewItem.ItemClass = searchPreviewItem[XmlElementNames_1.XmlElementNames.ItemClass];
            }
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.UniqueHash]) {
                previewItem.UniqueHash = searchPreviewItem[XmlElementNames_1.XmlElementNames.UniqueHash];
            }
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.SortValue]) {
                previewItem.SortValue = searchPreviewItem[XmlElementNames_1.XmlElementNames.SortValue];
            }
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.OwaLink]) {
                previewItem.OwaLink = searchPreviewItem[XmlElementNames_1.XmlElementNames.OwaLink];
            }
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.Sender]) {
                previewItem.Sender = searchPreviewItem[XmlElementNames_1.XmlElementNames.Sender];
            }
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.ToRecipients]) {
                previewItem.ToRecipients = this.GetRecipients(searchPreviewItem[XmlElementNames_1.XmlElementNames.ToRecipients]);
            }
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.CcRecipients]) {
                previewItem.CcRecipients = this.GetRecipients(searchPreviewItem[XmlElementNames_1.XmlElementNames.CcRecipients]);
            }
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.BccRecipients]) {
                previewItem.BccRecipients = this.GetRecipients(searchPreviewItem[XmlElementNames_1.XmlElementNames.BccRecipients]);
            }
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.CreatedTime]) {
                previewItem.CreatedTime = DateTime_1.DateTime.Parse(searchPreviewItem[XmlElementNames_1.XmlElementNames.CreatedTime]);
            }
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.ReceivedTime]) {
                previewItem.ReceivedTime = DateTime_1.DateTime.Parse(searchPreviewItem[XmlElementNames_1.XmlElementNames.ReceivedTime]);
            }
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.SentTime]) {
                previewItem.SentTime = DateTime_1.DateTime.Parse(searchPreviewItem[XmlElementNames_1.XmlElementNames.SentTime]);
            }
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.Subject]) {
                previewItem.Subject = searchPreviewItem[XmlElementNames_1.XmlElementNames.Subject];
            }
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.Preview]) {
                previewItem.Preview = searchPreviewItem[XmlElementNames_1.XmlElementNames.Preview];
            }
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.Size]) {
                previewItem.Size = ExtensionMethods_1.Convert.toNumber(searchPreviewItem[XmlElementNames_1.XmlElementNames.Size]);
            }
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.Importance]) {
                previewItem.Importance = Importance_1.Importance[searchPreviewItem[XmlElementNames_1.XmlElementNames.Importance]];
            }
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.Read]) {
                previewItem.Read = ExtensionMethods_1.Convert.toBool(searchPreviewItem[XmlElementNames_1.XmlElementNames.Read]);
            }
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.HasAttachment]) {
                previewItem.HasAttachment = ExtensionMethods_1.Convert.toBool(searchPreviewItem[XmlElementNames_1.XmlElementNames.HasAttachment]);
            }
            if (searchPreviewItem[XmlElementNames_1.XmlElementNames.ExtendedProperties]) {
                previewItem.ExtendedProperties = this.LoadExtendedPropertiesXmlJsObject(searchPreviewItem[XmlElementNames_1.XmlElementNames.ExtendedProperties], service);
            }
            previewItems.push(previewItem);
        }
        return previewItems.length === 0 ? null : previewItems;
    }
}
exports.SearchMailboxesResult = SearchMailboxesResult;
//# sourceMappingURL=SearchMailboxesResult.js.map