"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TimeWindow = void 0;
const ArgumentException_1 = require("../../Exceptions/ArgumentException");
const Strings_1 = require("../../Strings");
const XmlElementNames_1 = require("../../Core/XmlElementNames");
const XmlNamespace_1 = require("../../Enumerations/XmlNamespace");
/**
 * Represents a time period.
 *
 * @sealed
 */
class TimeWindow {
    constructor(startTime = null, endTime = null) {
        this.StartTime = startTime;
        this.EndTime = endTime;
    }
    /**
     * @internal Gets the duration.
     */
    get Duration() {
        return this.StartTime.Difference(this.EndTime);
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}				jsObject	Json Object converted from XML.
     * @param   {ExchangeService}	service	The service.
     */
    LoadFromXmlJsObject(jsonObject, service) {
        this.StartTime = service.ConvertUniversalDateTimeStringToLocalDateTime(jsonObject[XmlElementNames_1.XmlElementNames.StartTime]);
        this.EndTime = service.ConvertUniversalDateTimeStringToLocalDateTime(jsonObject[XmlElementNames_1.XmlElementNames.EndTime]);
    }
    /**
     * Validates this instance.
     *
     * ISelfValidate.Validate
     */
    Validate() {
        if (this.StartTime.CompareTo(this.EndTime) >= 0) {
            throw new ArgumentException_1.ArgumentException(Strings_1.Strings.TimeWindowStartTimeMustBeGreaterThanEndTime);
        }
    }
    /**
     * Writes to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer           The writer.
     * @param   {string}                xmlElementName   Name of the XML element.
     * @param   {any}                   startTime        The start time.
     * @param   {any}                   endTime          The end time.
     */
    static WriteToXml(writer, xmlElementName, startTime, endTime) {
        writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, xmlElementName);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.StartTime, startTime);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.EndTime, endTime);
        writer.WriteEndElement(); // xmlElementName
    }
    /**
     * @internal Writes to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer           The writer.
     * @param   {string}                xmlElementName   Name of the XML element.
     */
    WriteToXml(writer, xmlElementName) {
        TimeWindow.WriteToXml(writer, xmlElementName, this.StartTime, this.EndTime);
    }
    /**
     * @internal Writes to XML without scoping the dates and without emitting times.
     *
     * @param   {EwsServiceXmlWriter}   writer           The writer.
     * @param   {string}                xmlElementName   Name of the XML element.
     */
    WriteToXmlUnscopedDatesOnly(writer, xmlElementName) {
        const DateOnlyFormat = "YYYY-MM-DDT00:00:00";
        TimeWindow.WriteToXml(writer, xmlElementName, this.StartTime.Format(DateOnlyFormat), // CultureInfo.InvariantCulture),
        this.EndTime.Format(DateOnlyFormat) // CultureInfo.InvariantCulture));
        );
    }
}
exports.TimeWindow = TimeWindow;
//# sourceMappingURL=TimeWindow.js.map