"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MapiTypeConverter = void 0;
const AltDictionary_1 = require("../AltDictionary");
const Guid_1 = require("../Guid");
const DateTime_1 = require("../DateTime");
const ExtensionMethods_1 = require("../ExtensionMethods");
const EwsUtilities_1 = require("../Core/EwsUtilities");
const LazyMember_1 = require("../Core/LazyMember");
const ArgumentException_1 = require("../Exceptions/ArgumentException");
const MapiPropertyType_1 = require("../Enumerations/MapiPropertyType");
const MapiTypeConverterTypeSystem_1 = require("../Enumerations/MapiTypeConverterTypeSystem");
const MapiTypeConverterMapEntry_1 = require("./MapiTypeConverterMapEntry");
class MapiTypeConverter {
    static get MapiTypeConverterMap() { return MapiTypeConverter.mapiTypeConverterMap.Member; }
    static ChangeType(mapiType, value) {
        EwsUtilities_1.EwsUtilities.ValidateParam(value, "value");
        return MapiTypeConverter.MapiTypeConverterMap.get(mapiType).ChangeType(value);
    }
    static ConvertToString(mapiPropType, value) {
        return (value === null || value === undefined)
            ? ExtensionMethods_1.StringHelper.Empty
            : MapiTypeConverter.MapiTypeConverterMap.get(mapiPropType).ConvertToString(value);
    }
    static ConvertToValue(mapiPropType, strings) {
        if (typeof strings === 'string') {
            return MapiTypeConverter.MapiTypeConverterMap.get(mapiPropType).ConvertToValue(strings);
        }
        else if (Array.isArray(strings)) {
            EwsUtilities_1.EwsUtilities.ValidateParam(strings, "strings");
            var typeConverter = MapiTypeConverter.MapiTypeConverterMap.get(mapiPropType);
            var array = []; // = Array.CreateInstance(typeConverter.Type, strings.length);
            var index = 0;
            for (var stringValue of strings) {
                var value = typeConverter.ConvertToValueOrDefault(stringValue);
                array.push(value);
            }
            return array;
        }
        throw new ArgumentException_1.ArgumentException("parameter not in correct type");
    }
    static IsArrayType(mapiType) { return MapiTypeConverter.MapiTypeConverterMap.get(mapiType).IsArray; }
    static ParseMapiIntegerValue(s) {
        var num = ExtensionMethods_1.Convert.toNumber(s);
        if (num === NaN) {
            return s;
        }
        return num;
    }
}
exports.MapiTypeConverter = MapiTypeConverter;
MapiTypeConverter.UtcDataTimeStyles = DateTime_1.DateTimeStyles.AdjustToUniversal | DateTime_1.DateTimeStyles.AssumeUniversal;
MapiTypeConverter.mapiTypeConverterMap = new LazyMember_1.LazyMember(() => {
    var map = new AltDictionary_1.Dictionary((k) => k.toString());
    map.Add(MapiPropertyType_1.MapiPropertyType.ApplicationTime, new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.number, //double
    (s) => ExtensionMethods_1.Convert.toNumber(s), //Parse
    (o) => o.toString() //ConvertToString 
    ));
    map.Add(MapiPropertyType_1.MapiPropertyType.ApplicationTimeArray, new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.number, //double
    (s) => ExtensionMethods_1.Convert.toNumber(s), //Parse
    (o) => o.toString(), //ConvertToString
    true //IsArray
    ));
    var byteConverter = new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.byteArray, //byte[]
    (s) => ExtensionMethods_1.StringHelper.IsNullOrEmpty(s) ? null : ExtensionMethods_1.Convert.FromBase64String(s), //Parse
    (o) => ExtensionMethods_1.Convert.ToBase64String(o) //ConvertToString
    );
    map.Add(MapiPropertyType_1.MapiPropertyType.Binary, byteConverter);
    var byteArrayConverter = new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.byteArray, //byte[]
    (s) => ExtensionMethods_1.StringHelper.IsNullOrEmpty(s) ? null : ExtensionMethods_1.Convert.FromBase64String(s), //Parse
    (o) => ExtensionMethods_1.Convert.ToBase64String(o), //ConvertToString
    true //IsArray
    );
    map.Add(MapiPropertyType_1.MapiPropertyType.BinaryArray, byteArrayConverter);
    var boolConverter = new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.boolean, (s) => ExtensionMethods_1.Convert.toBool(s), //Parse
    (o) => o.toString().toLowerCase() //ConvertToString
    );
    map.Add(MapiPropertyType_1.MapiPropertyType.Boolean, boolConverter);
    var clsidConverter = new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.guid, (s) => new Guid_1.Guid(s), //Parse
    (o) => o.ToString() //ConvertToString
    );
    map.Add(MapiPropertyType_1.MapiPropertyType.CLSID, clsidConverter);
    var clsidArrayConverter = new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.guid, (s) => new Guid_1.Guid(s), //Parse
    (o) => o.ToString(), //ConvertToString
    true //IsArray
    );
    map.Add(MapiPropertyType_1.MapiPropertyType.CLSIDArray, clsidArrayConverter);
    map.Add(MapiPropertyType_1.MapiPropertyType.Currency, new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.number, //Int64
    (s) => ExtensionMethods_1.Convert.toNumber(s), //Parse
    (o) => o.toString() //ConvertToString 
    ));
    map.Add(MapiPropertyType_1.MapiPropertyType.CurrencyArray, new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.number, //Int64
    (s) => ExtensionMethods_1.Convert.toNumber(s), //Parse
    (o) => o.toString(), //ConvertToString
    true //IsArray
    ));
    map.Add(MapiPropertyType_1.MapiPropertyType.Double, new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.number, //double
    (s) => ExtensionMethods_1.Convert.toNumber(s), //Parse
    (o) => o.toString() //ConvertToString 
    ));
    map.Add(MapiPropertyType_1.MapiPropertyType.DoubleArray, new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.number, //double
    (s) => ExtensionMethods_1.Convert.toNumber(s), //Parse
    (o) => o.toString(), //ConvertToString
    true //IsArray
    ));
    map.Add(MapiPropertyType_1.MapiPropertyType.Error, new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.number, //Int32
    (s) => ExtensionMethods_1.Convert.toNumber(s), //Parse
    (o) => o.toString() //ConvertToString 
    ));
    map.Add(MapiPropertyType_1.MapiPropertyType.Float, new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.number, //float
    (s) => ExtensionMethods_1.Convert.toNumber(s), //Parse
    (o) => o.toString() //ConvertToString 
    ));
    map.Add(MapiPropertyType_1.MapiPropertyType.FloatArray, new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.number, //float
    (s) => ExtensionMethods_1.Convert.toNumber(s), //Parse
    (o) => o.toString(), //ConvertToString
    true //IsArray
    ));
    map.Add(MapiPropertyType_1.MapiPropertyType.Integer, new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.number, //Int32
    (s) => MapiTypeConverter.ParseMapiIntegerValue(s), //Parse
    (o) => o.toString() //ConvertToString 
    ));
    map.Add(MapiPropertyType_1.MapiPropertyType.IntegerArray, new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.number, //Int32
    (s) => ExtensionMethods_1.Convert.toNumber(s), //Parse //ref: check if latest ews managed api code changes this to same as Integer property type above
    (o) => o.toString(), //ConvertToString
    true //IsArray
    ));
    map.Add(MapiPropertyType_1.MapiPropertyType.Long, new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.number, //Int64
    (s) => ExtensionMethods_1.Convert.toNumber(s), //Parse
    (o) => o.toString() //ConvertToString 
    ));
    map.Add(MapiPropertyType_1.MapiPropertyType.LongArray, new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.number, //Int64
    (s) => ExtensionMethods_1.Convert.toNumber(s), //Parse
    (o) => o.toString(), //ConvertToString
    true //IsArray
    ));
    var objectConverter = new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.string, (s) => s, //Parse
    (o) => o.toString() //ConvertToString
    );
    map.Add(MapiPropertyType_1.MapiPropertyType.Object, objectConverter);
    var objectArrayConverter = new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.string, (s) => s, //Parse
    (o) => o.toString(), //ConvertToString
    true //IsArray
    );
    map.Add(MapiPropertyType_1.MapiPropertyType.ObjectArray, objectArrayConverter);
    map.Add(MapiPropertyType_1.MapiPropertyType.Short, new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.number, //short
    (s) => ExtensionMethods_1.Convert.toNumber(s), //Parse
    (o) => o.toString() //ConvertToString 
    ));
    map.Add(MapiPropertyType_1.MapiPropertyType.ShortArray, new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.number, //short
    (s) => ExtensionMethods_1.Convert.toNumber(s), //Parse
    (o) => o.toString(), //ConvertToString
    true //IsArray
    ));
    var stringConverter = new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.string, (s) => s, //Parse
    (o) => o.toString() //ConvertToString
    );
    map.Add(MapiPropertyType_1.MapiPropertyType.String, stringConverter);
    var stringArrayConverter = new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.string, (s) => s, //Parse
    (o) => o.toString(), //ConvertToString
    true //IsArray
    );
    map.Add(MapiPropertyType_1.MapiPropertyType.StringArray, stringArrayConverter);
    var sysTimeConverter = new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.DateTime, (s) => DateTime_1.DateTime.Parse(s), //Parse //ref: UtcDataTimeStyles not used, always utc, timezone not implemented
    (o) => EwsUtilities_1.EwsUtilities.DateTimeToXSDateTime(o) //ConvertToString
    );
    map.Add(MapiPropertyType_1.MapiPropertyType.SystemTime, sysTimeConverter);
    var sysTimeArrayConverter = new MapiTypeConverterMapEntry_1.MapiTypeConverterMapEntry(MapiTypeConverterTypeSystem_1.MapiTypeConverterTypeSystem.DateTime, (s) => DateTime_1.DateTime.Parse(s), //Parse //ref: UtcDataTimeStyles not used, always utc, timezone not implemented
    (o) => EwsUtilities_1.EwsUtilities.DateTimeToXSDateTime(o), //ConvertToString
    true //IsArray
    );
    map.Add(MapiPropertyType_1.MapiPropertyType.SystemTimeArray, sysTimeArrayConverter);
    return map;
});
//# sourceMappingURL=MapiTypeConverter.js.map