"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FolderEvent = void 0;
const ExtensionMethods_1 = require("../ExtensionMethods");
const EventType_1 = require("../Enumerations/EventType");
const FolderId_1 = require("../ComplexProperties/FolderId");
const JsonNames_1 = require("../Core/JsonNames");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const NotificationEvent_1 = require("./NotificationEvent");
/**
 * Represents an event that applies to a folder.
 */
class FolderEvent extends NotificationEvent_1.NotificationEvent {
    /**
     * @internal Initializes a new instance of the **FolderEvent** class.
     *
     * @param   {EventType}		eventType   Type of the event.
     * @param   {DateTime}   	timestamp   The event timestamp.
     */
    constructor(eventType, timestamp) {
        super(eventType, timestamp);
        this.folderId = null;
        this.oldFolderId = null;
        /**
         * The new number of unread messages. This is is only meaningful when EventType is equal to EventType.Modified. For all other event types, it's null.
         */
        this.unreadCount = null;
    }
    /**
     * Gets the Id of the folder this event applies to.
     */
    get FolderId() {
        return this.folderId;
    }
    /**
     * Gets the Id of the folder that was moved or copied. OldFolderId is only meaningful when EventType is equal to either EventType.Moved or EventType.Copied. For all other event types, OldFolderId is null.
     */
    get OldFolderId() {
        return this.oldFolderId;
    }
    /**
     * Gets the new number of unread messages. This is is only meaningful when EventType is equal to EventType.Modified. For all other event types, UnreadCount is null.
     */
    get UnreadCount() {
        return this.unreadCount;
    }
    /**
     * @internal Loads from XML.
     *
     * @param   {any}                 jsEvent                Json Object converted from XML.
     * @param   {ExchangeService}     service                The service.
     */
    LoadFromXmlJsObject(jsEvent, service) {
        this.folderId = new FolderId_1.FolderId();
        this.folderId.LoadFromXmlJsObject(jsEvent[XmlElementNames_1.XmlElementNames.FolderId], service);
        this.ParentFolderId = new FolderId_1.FolderId();
        this.ParentFolderId.LoadFromXmlJsObject(jsEvent[XmlElementNames_1.XmlElementNames.ParentFolderId], service);
        switch (this.EventType) {
            case EventType_1.EventType.Moved:
            case EventType_1.EventType.Copied:
                this.oldFolderId = new FolderId_1.FolderId();
                this.oldFolderId.LoadFromXmlJsObject(jsEvent[JsonNames_1.JsonNames.OldFolderId], service);
                this.OldParentFolderId = new FolderId_1.FolderId();
                this.OldParentFolderId.LoadFromXmlJsObject(jsEvent[XmlElementNames_1.XmlElementNames.OldParentFolderId], service);
                break;
            case EventType_1.EventType.Modified:
                if (jsEvent[XmlElementNames_1.XmlElementNames.UnreadCount]) {
                    this.unreadCount = ExtensionMethods_1.Convert.toNumber(jsEvent[XmlElementNames_1.XmlElementNames.UnreadCount]);
                }
                break;
            default:
                break;
        }
    }
}
exports.FolderEvent = FolderEvent;
//# sourceMappingURL=FolderEvent.js.map