"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GetEventsResults = void 0;
const ExtensionMethods_1 = require("../ExtensionMethods");
const AltDictionary_1 = require("../AltDictionary");
const EventType_1 = require("../Enumerations/EventType");
const EwsServiceJsonReader_1 = require("../Core/EwsServiceJsonReader");
const FolderEvent_1 = require("./FolderEvent");
const ItemEvent_1 = require("./ItemEvent");
const LazyMember_1 = require("../Core/LazyMember");
const XmlElementNames_1 = require("../Core/XmlElementNames");
/**
 * Represents a collection of notification events.
 */
class GetEventsResults {
    /**
     * @internal Initializes a new instance of the **GetEventsResults** class.
     */
    constructor() {
        /**
         * Watermark in event.
         */
        this.newWatermark = null;
        /**
         * Subscription id.
         */
        this.subscriptionId = null;
        /**
         * Previous watermark.
         */
        this.previousWatermark = null;
        /**
         * True if more events available for this subscription.
         */
        this.moreEventsAvailable = false;
        /**
         * Collection of notification events.
         */
        this.events = [];
    }
    /**
     * Gets the XML element name to event type mapping.
     *
     * @value	The XML element name to event type mapping.
     */
    static get XmlElementNameToEventTypeMap() {
        return GetEventsResults.xmlElementNameToEventTypeMap.Member;
    }
    /**
     * @internal Gets the Id of the subscription the collection is associated with.
     */
    get SubscriptionId() {
        return this.subscriptionId;
    }
    /**
     * @internal Gets the subscription's previous watermark.
     */
    get PreviousWatermark() {
        return this.previousWatermark;
    }
    /**
     * @internal Gets the subscription's new watermark.
     */
    get NewWatermark() {
        return this.newWatermark;
    }
    /**
     * @internal Gets a value indicating whether more events are available on the Exchange server.
     */
    get MoreEventsAvailable() {
        return this.moreEventsAvailable;
    }
    /**
     * Gets the collection of folder events.
     *
     * @value	The folder events.
     */
    get FolderEvents() {
        return ExtensionMethods_1.ArrayHelper.OfType(this.events, (item) => item instanceof FolderEvent_1.FolderEvent);
    }
    /**
     * Gets the collection of item events.
     *
     * @value	The item events.
     */
    get ItemEvents() {
        return ExtensionMethods_1.ArrayHelper.OfType(this.events, (item) => item instanceof ItemEvent_1.ItemEvent);
    }
    /**
     * Gets the collection of all events.
     *
     * @value	The events.
     */
    get AllEvents() {
        return this.events;
    }
    /**
     * Loads the events from XML.
     *
     * @param   {any[]}               jsEventsArray         The json events array.
     * @param   {string}     		  xmlElementName		Name of the element.
     * @param   {ExchangeService}     service               The service.
     */
    LoadEventsFromXmlJsObject(jsEventsArray, eventElementName, service) {
        for (let jsEvent of jsEventsArray) {
            this.newWatermark = jsEvent[XmlElementNames_1.XmlElementNames.Watermark];
            let eventType = GetEventsResults.XmlElementNameToEventTypeMap.get(eventElementName);
            if (eventType == EventType_1.EventType.Status) {
                continue;
            }
            let timeStamp = service.ConvertUniversalDateTimeStringToLocalDateTime(jsEvent[XmlElementNames_1.XmlElementNames.TimeStamp]);
            let notificationEvent;
            if (jsEvent[XmlElementNames_1.XmlElementNames.FolderId]) {
                notificationEvent = new FolderEvent_1.FolderEvent(eventType, timeStamp);
            }
            else {
                notificationEvent = new ItemEvent_1.ItemEvent(eventType, timeStamp);
            }
            notificationEvent.LoadFromXmlJsObject(jsEvent, service);
            this.events.push(notificationEvent);
        }
    }
    /**
     * @internal Loads from XML.
     *
     * @param   {any}                 eventsResponse         The events response Object converted from XML.
     * @param   {ExchangeService}     service                The service.
     */
    LoadFromXmlJsObject(eventsResponse, service) {
        let response = eventsResponse;
        if (eventsResponse[XmlElementNames_1.XmlElementNames.Notification]) {
            response = eventsResponse[XmlElementNames_1.XmlElementNames.Notification];
        }
        for (let key in response) {
            switch (key) {
                case XmlElementNames_1.XmlElementNames.SubscriptionId:
                    this.subscriptionId = response[key];
                    break;
                case XmlElementNames_1.XmlElementNames.PreviousWatermark:
                    this.previousWatermark = response[key];
                    break;
                case XmlElementNames_1.XmlElementNames.MoreEvents:
                    this.moreEventsAvailable = ExtensionMethods_1.Convert.toBool(response[key]);
                    break;
                default:
                    if (GetEventsResults.XmlElementNameToEventTypeMap.containsKey(key)) {
                        this.LoadEventsFromXmlJsObject(EwsServiceJsonReader_1.EwsServiceJsonReader.ReadAsArray(response, key), key, service);
                    }
                    break;
            }
        }
    }
}
exports.GetEventsResults = GetEventsResults;
/**
 * Map XML element name to notification event type.
 *
 * /remarks/ 	If you add a new notification event type, you'll need to add a new entry to the dictionary here.
 */
GetEventsResults.xmlElementNameToEventTypeMap = new LazyMember_1.LazyMember(() => {
    var dictionary = new AltDictionary_1.DictionaryWithStringKey();
    dictionary.Add("CopiedEvent", EventType_1.EventType.Copied);
    dictionary.Add("CreatedEvent", EventType_1.EventType.Created);
    dictionary.Add("DeletedEvent", EventType_1.EventType.Deleted);
    dictionary.Add("ModifiedEvent", EventType_1.EventType.Modified);
    dictionary.Add("MovedEvent", EventType_1.EventType.Moved);
    dictionary.Add("NewMailEvent", EventType_1.EventType.NewMail);
    dictionary.Add("StatusEvent", EventType_1.EventType.Status);
    dictionary.Add("FreeBusyChangedEvent", EventType_1.EventType.FreeBusyChanged);
    return dictionary;
});
//# sourceMappingURL=GetEventsResults.js.map