"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DateTimePropertyDefinition = void 0;
const DateTime_1 = require("../DateTime");
const EwsLogging_1 = require("../Core/EwsLogging");
const EwsUtilities_1 = require("../Core/EwsUtilities");
const PropertyException_1 = require("../Exceptions/PropertyException");
const ExtensionMethods_1 = require("../ExtensionMethods");
const Strings_1 = require("../Strings");
const TimeZoneInfo_1 = require("../TimeZoneInfo");
const XmlNamespace_1 = require("../Enumerations/XmlNamespace");
const PropertyDefinition_1 = require("./PropertyDefinition");
/**
 * @internal Represents DateTime property definition.
 */
class DateTimePropertyDefinition extends PropertyDefinition_1.PropertyDefinition {
    constructor(propertyName, xmlElementName, uri, versionOrFlags, version, isNullable) {
        switch (arguments.length) {
            case 4:
                super(propertyName, xmlElementName, uri, versionOrFlags);
                break;
            case 5:
            case 6:
                super(propertyName, xmlElementName, uri, versionOrFlags, version);
                break;
            default:
                break;
        }
        // @ts-ignore
        this.isNullable = isNullable || false;
    }
    /**
     * @internal Gets a value indicating whether this property definition is for a nullable type (ref, int?, bool?...).
     */
    get IsNullable() { return this.isNullable; }
    /**
     * Gets the converted date time.
     *
     * @param   {ExchangeServiceBase}   service             The service.
     * @param   {PropertyBag}           propertyBag         The property bag.
     * @param   {boolean}               isUpdateOperation   if set to true [is update operation].
     * @param   {any}                   value               The value.
     * @return  {DateTime}      Converted DateTime value
     */
    GetConvertedDateTime(service, propertyBag, isUpdateOperation, value) {
        EwsLogging_1.EwsLogging.Assert(false, "DateTimePropertyDefinition.GetConvertedDateTime", "TimeZone info could be misleading, It should be used as UTC in all cases until fixed");
        var dateTime = DateTime_1.DateTime.Parse(value);
        var convertedDateTime;
        //debug: //todo: find datetimekind
        // If the date/time is unspecified, we may need to scope it to time zone.
        if (dateTime.Kind == DateTime_1.DateTimeKind.Unspecified) {
            convertedDateTime = this.ScopeToTimeZone(service, value, propertyBag, isUpdateOperation);
        }
        else {
            convertedDateTime = dateTime;
        }
        return convertedDateTime;
    }
    /**
     * @internal Loads the property value from XMLJsObject.
     *
     * @param   {any}               value         The JSON value.  Can be a JsonObject, string, number, bool, array, or null.
     * @param   {ExchangeService}   service       The service.
     * @param   {PropertyBag}       propertyBag   The property bag.
     */
    LoadPropertyValueFromXmlJsObject(jsObject, service, propertyBag) {
        var stringValue = jsObject; //.toString();
        //debug: //ref: check for datetime value
        if (!ExtensionMethods_1.StringHelper.IsNullOrEmpty(stringValue)) {
            var value = service.ConvertUniversalDateTimeStringToLocalDateTime(stringValue);
            propertyBag._setItem(this, service.ConvertUniversalDateTimeStringToLocalDateTime(stringValue));
        }
    }
    /**
     * @internal Scopes the date time property to the appropriate time zone, if necessary.
     *
     * @param   {ExchangeServiceBase}   service             The service emitting the request.
     * @param   {DateTime}              dateTime            The date time.
     * @param   {PropertyBag}           propertyBag         The property bag.
     * @param   {boolean}               isUpdateOperation   Indicates whether the scoping is to be performed in the context of an update operation.
     * @return  {DateTime}              The converted DateTime.
     */
    ScopeToTimeZone(service, dateTime, propertyBag, isUpdateOperation) {
        EwsLogging_1.EwsLogging.Assert(false, "DateTimePropertyDefinition.ScopeToTimeZone", "TimeZone info could be misleading, It should be used as UTC in all cases until fixed");
        try {
            var convertedDateTime = EwsUtilities_1.EwsUtilities.ConvertTime(dateTime, service.TimeZone, TimeZoneInfo_1.TimeZoneInfo.Utc);
            return new DateTime_1.DateTime(convertedDateTime.TotalMilliSeconds, DateTime_1.DateTimeKind.Utc);
        }
        catch (e) //TimeZoneConversionException
         {
            throw new PropertyException_1.PropertyException(ExtensionMethods_1.StringHelper.Format(Strings_1.Strings.InvalidDateTime, dateTime), this.Name, e);
        }
    }
    /**
     * @internal Writes the property value to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer              The writer.
     * @param   {PropertyBag}           propertyBag         The property bag.
     * @param   {boolean}               isUpdateOperation   Indicates whether the context is an update operation.
     */
    WritePropertyValueToXml(writer, propertyBag, isUpdateOperation) {
        var value = propertyBag._getItem(this);
        if (value != null) {
            writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, this.XmlElementName);
            var convertedDateTime = this.GetConvertedDateTime(writer.Service, propertyBag, isUpdateOperation, value);
            writer.WriteValue(EwsUtilities_1.EwsUtilities.DateTimeToXSDateTime(convertedDateTime), this.Name);
            writer.WriteEndElement();
        }
    }
}
exports.DateTimePropertyDefinition = DateTimePropertyDefinition;
//# sourceMappingURL=DateTimePropertyDefinition.js.map