"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SearchFilter = void 0;
const LogicalOperator_1 = require("../../Enumerations/LogicalOperator");
const ExtensionMethods_1 = require("../../ExtensionMethods");
const XmlElementNames_1 = require("../../Core/XmlElementNames");
const ComplexProperty_1 = require("../../ComplexProperties/ComplexProperty");
/**
 * Represents the base search filter class. Use descendant search filter classes such as SearchFilter.IsEqualTo, SearchFilter.ContainsSubstring and SearchFilter.SearchFilterCollection to define search filters.
 */
class SearchFilter extends ComplexProperty_1.ComplexProperty {
    /**
     * Initializes a new instance of the **SearchFilter** class.
     */
    constructor() {
        super();
    }
    /**
     * Gets the search filter instance.
     *
     * @param {string} localName Name of the local.
     * @returns {SearchFilter}
     */
    static GetSearchFilterInstance(localName) {
        let searchFilter;
        switch (localName) {
            case XmlElementNames_1.XmlElementNames.Exists:
                searchFilter = new SearchFilter.Exists();
                break;
            case XmlElementNames_1.XmlElementNames.Contains:
                searchFilter = new SearchFilter.ContainsSubstring();
                break;
            case XmlElementNames_1.XmlElementNames.Excludes:
                searchFilter = new SearchFilter.ExcludesBitmask();
                break;
            case XmlElementNames_1.XmlElementNames.Not:
                searchFilter = new SearchFilter.Not();
                break;
            case XmlElementNames_1.XmlElementNames.And:
                searchFilter = new SearchFilter.SearchFilterCollection(LogicalOperator_1.LogicalOperator.And);
                break;
            case XmlElementNames_1.XmlElementNames.Or:
                searchFilter = new SearchFilter.SearchFilterCollection(LogicalOperator_1.LogicalOperator.Or);
                break;
            case XmlElementNames_1.XmlElementNames.IsEqualTo:
                searchFilter = new SearchFilter.IsEqualTo();
                break;
            case XmlElementNames_1.XmlElementNames.IsNotEqualTo:
                searchFilter = new SearchFilter.IsNotEqualTo();
                break;
            case XmlElementNames_1.XmlElementNames.IsGreaterThan:
                searchFilter = new SearchFilter.IsGreaterThan();
                break;
            case XmlElementNames_1.XmlElementNames.IsGreaterThanOrEqualTo:
                searchFilter = new SearchFilter.IsGreaterThanOrEqualTo();
                break;
            case XmlElementNames_1.XmlElementNames.IsLessThan:
                searchFilter = new SearchFilter.IsLessThan();
                break;
            case XmlElementNames_1.XmlElementNames.IsLessThanOrEqualTo:
                searchFilter = new SearchFilter.IsLessThanOrEqualTo();
                break;
            default:
                searchFilter = null;
                break;
        }
        return searchFilter;
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}               jsObject                Json Object converted from XML.
     * @param   {ExchangeService}   service                 The service.
     * @param   {string}            typeName                 type name, when provided prevent call for type name check.
     * @return  {SearchFilter}      SearchFilter instance.
     */
    static LoadFromXmlJsObject(jsObject, service, typeName = null) {
        if (!typeName) {
            typeName = ExtensionMethods_1.TypeSystem.GetJsObjectTypeName(jsObject);
        }
        let searchFilter = SearchFilter.GetSearchFilterInstance(jsObject.ReadTypeString());
        if (searchFilter != null) {
            searchFilter.LoadFromXmlJsObject(jsObject, service);
        }
        return searchFilter;
    }
    /**
     * @internal Writes to XML.
     *
     * @param {EwsServiceXmlWriter} writer  The writer.
     */
    WriteToXml(writer) {
        super.WriteToXml(writer, this.GetXmlElementName());
    }
}
exports.SearchFilter = SearchFilter;
(function (SearchFilter) {
})(SearchFilter = exports.SearchFilter || (exports.SearchFilter = {}));
//# sourceMappingURL=SearchFilter.js.map