"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ChangeCollection = void 0;
const ArgumentException_1 = require("../Exceptions/ArgumentException");
const EwsLogging_1 = require("../Core/EwsLogging");
const Strings_1 = require("../Strings");
/**
 * Represents a collection of changes as returned by a synchronization operation.
 *
 * @sealed
 * @typeparam	{TChange}	Type representing the type of change (e.g. FolderChange or ItemChange)
 */
class ChangeCollection {
    /**
     * @internal Initializes a new instance of the **ChangeCollection<TChange>** class.
     */
    constructor() {
        this.___typeGenerics = ["Change"];
        this.changes = [];
        this.syncState = null;
        this.moreChangesAvailable = false;
    }
    /**
     * Gets the number of changes in the collection.
     */
    get Count() {
        return this.changes.length;
    }
    /**
     * Gets the SyncState blob returned by a synchronization operation.
     */
    get SyncState() {
        return this.syncState;
    }
    set SyncState(value) {
        this.syncState = value;
    }
    /**
     * Gets a value indicating whether the there are more changes to be synchronized from the server.
     */
    get MoreChangesAvailable() {
        return this.moreChangesAvailable;
    }
    set MoreChangesAvailable(value) {
        this.moreChangesAvailable = value;
    }
    /**
     * Gets an individual change from the change collection.
     *
     * @param   {number}   index   Zero-based index.
     * @return  {TChange}		An single change.
     */
    _getItem(index) {
        if (index < 0 || index >= this.Count) {
            throw new ArgumentException_1.ArgumentOutOfRangeException("index", Strings_1.Strings.IndexIsOutOfRange);
        }
        return this.changes[index];
    }
    /**
     * @internal Adds the specified change.
     *
     * @param   {TChange}   change   The change.
     */
    Add(change) {
        EwsLogging_1.EwsLogging.Assert(change != null, "ChangeList.Add", "change is null");
        this.changes.push(change);
    }
    /**
     *  Returns an enumerator that iterates through the collection. this case this.changes
     */
    GetEnumerator() {
        return this.changes;
    }
}
exports.ChangeCollection = ChangeCollection;
//# sourceMappingURL=ChangeCollection.js.map