"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.XHRDefault = void 0;
const fetch_1 = require("fetch");
/**
 * Default implementation of XHRApi using fetch
 */
class XHRDefault {
    constructor(fetchOptions = {}) {
        this.fetchOptions = {};
        this.fetchOptions = fetchOptions;
        try {
            let fetch = require("fetch");
            XHRDefault.FetchStream = fetch.FetchStream;
            XHRDefault.fetchUrl = fetch.fetchUrl;
        }
        catch (e) { }
    }
    xhr(xhroptions, progressDelegate) {
        if (XHRDefault.fetchUrl === null) {
            throw new Error("xhrApi - stub method, must be bootstrapped");
        }
        //setup xhr for github.com/andris9/fetch options
        let options = {
            url: xhroptions.url,
            payload: xhroptions.data,
            headers: xhroptions.headers,
            method: xhroptions.type,
            disableRedirects: !xhroptions.allowRedirect,
        };
        return new Promise((resolve, reject) => {
            XHRDefault.fetchUrl(xhroptions.url, this.getOptions(options), (error, meta, body) => {
                if (error) {
                    if (typeof error.status === 'undefined') {
                        error.status = 0;
                    }
                    reject(setupXhrResponse(error));
                }
                else {
                    let xhrResponse = {
                        response: body.toString(),
                        status: meta.status,
                        redirectCount: meta.redirectCount,
                        headers: meta.responseHeaders,
                        finalUrl: meta.finalUrl,
                        responseType: '',
                        statusText: undefined,
                    };
                    if (xhrResponse.status === 200) {
                        resolve(setupXhrResponse(xhrResponse));
                    }
                    else {
                        reject(setupXhrResponse(xhrResponse));
                    }
                }
            });
        });
    }
    xhrStream(xhroptions, progressDelegate) {
        if (XHRDefault.FetchStream === null) {
            throw new Error("xhrApi - stub method, must be bootstrapped");
        }
        //setup xhr for github.com/andris9/fetch options
        let options = {
            payload: xhroptions.data,
            headers: xhroptions.headers,
            method: xhroptions.type
        };
        return new Promise((resolve, reject) => {
            this.stream = new XHRDefault.FetchStream(xhroptions.url, this.getOptions(options));
            this.stream.on("data", (chunk) => {
                //console.log(chunk.toString());
                progressDelegate({ type: "data", data: chunk.toString() });
            });
            this.stream.on("meta", (meta) => {
                progressDelegate({ type: "header", headers: meta["responseHeaders"] });
            });
            this.stream.on("end", () => {
                progressDelegate({ type: "end" });
                resolve(void 0);
            });
            this.stream.on('error', (error) => {
                progressDelegate({ type: "error", error: error });
                this.disconnect();
                reject(error);
            });
        });
    }
    disconnect() {
        if (this.stream) {
            try {
                this.stream.destroy();
            }
            catch (e) { }
        }
    }
    get apiName() {
        return "default";
    }
    getOptions(opts) {
        let headers = Object.assign({}, (XHRDefault.defaultOptions || {}).headers, (this.fetchOptions || {}).headers, (opts || {}).headers);
        return Object.assign({}, XHRDefault.defaultOptions, this.fetchOptions, opts, { headers });
    }
}
exports.XHRDefault = XHRDefault;
XHRDefault.FetchStream = fetch_1.FetchStream;
XHRDefault.fetchUrl = null;
XHRDefault.defaultOptions = {};
/** @internal */
function setupXhrResponse(xhrResponse) {
    xhrResponse["responseText"] = xhrResponse.response;
    delete xhrResponse["response"];
    xhrResponse.getAllResponseHeaders = function () {
        var header = "";
        if (xhrResponse.headers) {
            for (var key in xhrResponse.headers) {
                header += key + " : " + xhrResponse.headers[key] + "\r\n";
            }
        }
        return header;
    };
    xhrResponse.getResponseHeader = (header) => {
        if (header) {
            if (xhrResponse.headers) {
                if (xhrResponse.headers[header]) {
                    return xhrResponse.headers[header];
                }
                if (xhrResponse.headers[header.toLocaleLowerCase()]) {
                    return xhrResponse.headers[header.toLocaleLowerCase()];
                }
            }
        }
        return null;
    };
    return xhrResponse;
}
//# sourceMappingURL=XHRDefault.js.map